package nintaco.gui.glasses;

import java.util.Arrays;
import nintaco.*;
import nintaco.gui.*;
import nintaco.input.*;
import nintaco.input.glasses.*;
import nintaco.mappers.*;
import nintaco.palettes.*;
import nintaco.preferences.*;
import nintaco.tv.*;
import static nintaco.util.GuiUtil.*;
import static nintaco.input.InputDevices.*;
import static nintaco.tv.TVSystem.*;

public class GlassesFrame extends javax.swing.JFrame {

  private static final int CLEAR_TIMER_RESET_VALUE = 3;
  
  private final ImagePanel imagePanel = new ImagePanel(512, 240);
  
  private boolean exchange;
  private boolean lastShutter;
  private int clearTimer = CLEAR_TIMER_RESET_VALUE;
  
  public GlassesFrame() {
    initComponents();
    setContentPane(imagePanel);
    scaleFonts(this);
    pack();
    moveToImageFrameMonitor(this);
    setExchange(AppPrefs.getInstance().getInputs().isExchangeGlasses());
  }
  
  public final void setExchange(final boolean exchange) {
    this.exchange = exchange;
    clear();
  }
  
  public void clear() {
    final ImagePanel panel = (ImagePanel)imagePanel;
    Arrays.fill(panel.getScreen(), 0);
    panel.render();    
  }
  
  public void update(final int[] screen) {
    lastShutter ^= true;
    render(NTSC, screen, (lastShutter ^ exchange) ? 255 : 511);
  }
  
  public void update(final MachineRunner machineRunner) {
    
    if (--clearTimer == 0) {
      clear();
    }
    
    final PPU ppu = machineRunner.getPPU();
    final Mapper mapper = machineRunner.getMapper();
    if (ppu == null || mapper == null) {
      return;
    }
    
    final DeviceMapper[] deviceMappers = mapper.getDeviceMappers();
    if (deviceMappers == null || deviceMappers.length == 0) {
      return;
    }
    final int offsetX;
    outer: {
      for(int i = deviceMappers.length - 1; i >= 0; i--) {
        if (deviceMappers[i].getInputDevice() == Glasses) {
          final boolean shutter = ((GlassesMapper)deviceMappers[i]).isShutter();
          if (shutter != lastShutter) {
            lastShutter = shutter;
            clearTimer = CLEAR_TIMER_RESET_VALUE;
          }
          offsetX = (shutter ^ exchange) ? 511 : 255;
          break outer;
        }
      }
      return;
    }
    
    render(mapper.getTVSystem(), ppu.getLastScreen(), offsetX);
  }
  
  private void render(final TVSystem tvSystem, final int[] screen, 
      final int offsetX) {
    
    final ImagePanel panel = (ImagePanel)imagePanel;
    final int[] palette = PaletteUtil.getExtendedPalette(tvSystem);    
    final int[] s = panel.getScreen();
    for(int y = 239; y >= 0; y--) {
      final int sy = y << 8;
      final int dy = y << 9;
      for(int sx = 255, dx = offsetX; sx >= 0; sx--, dx--) {
        s[dy | dx] = palette[screen[sy | sx]];
      }      
    }
    panel.render();
  }
  
  public void destroy() {
    dispose();
  }
  
  private void closeFrame() {
    App.destroyGlassesFrame();
  }

  /**
   * This method is called from within the constructor to initialize the form.
   * WARNING: Do NOT modify this code. The content of this method is always
   * regenerated by the Form Editor.
   */
  @SuppressWarnings("unchecked")
  // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
  private void initComponents() {

    setDefaultCloseOperation(javax.swing.WindowConstants.DO_NOTHING_ON_CLOSE);
    setTitle("Famicom 3D Glasses");
    setMaximumSize(null);
    setMinimumSize(null);
    setPreferredSize(null);
    addWindowListener(new java.awt.event.WindowAdapter() {
      public void windowClosing(java.awt.event.WindowEvent evt) {
        formWindowClosing(evt);
      }
    });

    javax.swing.GroupLayout layout = new javax.swing.GroupLayout(getContentPane());
    getContentPane().setLayout(layout);
    layout.setHorizontalGroup(
      layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
      .addGap(0, 591, Short.MAX_VALUE)
    );
    layout.setVerticalGroup(
      layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
      .addGap(0, 285, Short.MAX_VALUE)
    );

    pack();
  }// </editor-fold>//GEN-END:initComponents

  private void formWindowClosing(java.awt.event.WindowEvent evt) {//GEN-FIRST:event_formWindowClosing
    closeFrame();
  }//GEN-LAST:event_formWindowClosing

  // Variables declaration - do not modify//GEN-BEGIN:variables
  // End of variables declaration//GEN-END:variables
}
