package nintaco.api.local;

import java.awt.*;
import java.awt.image.*;
import java.io.*;
import java.util.*;
import java.util.List;
import java.util.concurrent.*;
import nintaco.*;
import nintaco.api.*;
import nintaco.cheats.*;
import nintaco.files.*;
import nintaco.gui.fonts.*;
import nintaco.input.*;
import nintaco.input.other.*;
import nintaco.input.zapper.*;
import nintaco.mappers.*;
import nintaco.mappers.nintendo.vs.*;
import nintaco.palettes.*;
import nintaco.preferences.*;
import static java.awt.image.BufferedImage.*;
import static java.lang.Math.*;
import static nintaco.api.local.EventTypes.*;
import static nintaco.gui.image.ImagePane.*;
import static nintaco.util.BitUtil.*;
import static nintaco.util.CollectionsUtil.*;
import static nintaco.util.StringUtil.*;
import static nintaco.tv.TVSystem.*;

public class LocalAPI implements API {
  
  private static volatile LocalAPI localAPI;
  
  public static void setLocalAPI(final LocalAPI localAPI) {
    LocalAPI.localAPI = localAPI;
  }
  
  public static LocalAPI getLocalAPI() {
    return localAPI;
  }  
  
  private final Map<Integer, Sprite> sprites = new ConcurrentHashMap<>();
  
  private AccessPoint[] accessPoints;
  private ControllersListener[] controllersListeners;
  private FrameListener[] frameListeners;
  private ScanlinePoint[] scanlinePoints;
  private ActivateListener[] activateListeners;
  private DeactivateListener[] deactivateListeners;
  private SpriteZeroListener[] spriteZeroListeners;
  private StopListener[] stopListeners;
  private ScanlineCyclePoint[][][] scanlineCyclePoints;
  private StatusListener[] statusListeners;
  
  private final BufferedImage image;
  private final Graphics2D g;
  private final int[] pixels;
  
  private boolean pixelsModified;
  private boolean disposed;
  
  private int requestColor = Colors.WHITE;

  private int buttons;
  private int buttonsAND = -1;
  private int buttonsOR;
  
  private volatile MachineRunner _machineRunner;
  private volatile CPU _cpu;
  private volatile PPU _ppu;
  private volatile Mapper _mapper;
  private volatile int[] screen;
  private volatile boolean stopped = true;  
  
  public LocalAPI() {
    image = new BufferedImage(IMAGE_WIDTH, IMAGE_HEIGHT, TYPE_INT_ARGB);
    g = image.createGraphics();
    pixels = ((DataBufferInt)image.getRaster().getDataBuffer()).getData();
    for(int i = pixels.length - 1; i >= 0; i--) {
      pixels[i] = -1;
    }
  }
  
  @Override
  @ApiMethod(value = 0, autogenerated = false)
  public void run() {
    App.setLocalAPI(this);
  }
  
  public void dispose() {
    
    synchronized(this) {
      if (disposed) {
        return;
      } else {
        disposed = true;
      }
    }
    
    final StopListener[] listeners = stopListeners;
    if (listeners != null) {
      for(int i = listeners.length - 1; i >= 0; i--) {
        listeners[i].dispose();
      }
    }
    
    g.dispose();
    
    App.setLocalAPI(null);
    final CPU cpu = _cpu;
    if (cpu != null) {
      cpu.setAccessPoints(null);
    }
    final PPU ppu = _ppu;
    if (ppu != null) {
      ppu.clearLocalAPI();
    }
    final MachineRunner machineRunner = _machineRunner;
    if (machineRunner != null) {
      machineRunner.clearLocalAPI();
    }
  }
  
  public synchronized void setMachineRunner(final MachineRunner machineRunner) {
    if (machineRunner == null) {
      _machineRunner = null;
      _cpu = null;
      _ppu = null;
      _mapper = null;
      screen = null;            
    } else {
      _machineRunner = machineRunner;
      _cpu = machineRunner.getCPU();
      _ppu = machineRunner.getPPU();
      _mapper = machineRunner.getMapper();
      
      _machineRunner.setLocalAPI(this);
    }
  }
  
  public void frameRendered(final int[] screen) {
    this.screen = screen;
    final FrameListener[] listeners = frameListeners;
    if (listeners != null) {
      for(int i = listeners.length - 1; i >= 0; i--) {
        listeners[i].frameRendered();        
      }
    }
    final int[] s = this.screen;
    if (pixelsModified) {
      pixelsModified = false;
      if (s == null) {
        for(int i = pixels.length - 1; i >= 0; i--) {
          pixels[i] = -1;
        }
      } else {
        for(int i = s.length - 1; i >= 0; i--) {
          if (pixels[i] != -1) {
            s[i] = pixels[i] & 0x1FF;
            pixels[i] = -1;
          }
        }
      }
    }    
  }
  
  public void scanlineRendered(final int scanline) {
    final ScanlinePoint[] points = scanlinePoints;
    if (points != null) {
      for(int i = points.length - 1; i >= 0; i--) {
        final ScanlinePoint point = scanlinePoints[i];
        if (point.scanline == scanline) {
          point.listener.scanlineRendered(scanline);
        }
      }
    }
  }
  
  public void cyclePerformed(final int scanline, final int scanlineCycle, 
      final int address, final boolean rendering) {
    
    final ScanlineCyclePoint[][][] matrix = scanlineCyclePoints;
    if (matrix != null) {
      final ScanlineCyclePoint[][] row = matrix[scanline + 1];
      if (row != null) {
        final ScanlineCyclePoint[] points = row[scanlineCycle];
        if (points != null) {
          for(int i = points.length - 1; i >= 0; i--) {
            points[i].listener.cyclePerformed(scanline, scanlineCycle, address,
                rendering);
          }
        }
      }
    }
  }
  
  public int controllersProbed(final int buttons) {

    if (stopped) {
      final PPU ppu = _ppu;
      final CPU cpu = _cpu;
      if (ppu != null && cpu != null) {
        stopped = false;
        ppu.setLocalAPI(this);
        cpu.setAccessPoints(accessPoints);
        final ActivateListener[] listeners = activateListeners;
        if (listeners != null) {
          for(int i = listeners.length - 1; i >= 0; i--) {
            listeners[i].apiEnabled();
          }
        }        
      }
    }
    
    this.buttons = buttons;
    final ControllersListener[] listeners = controllersListeners;
    if (listeners != null) {
      for(int i = listeners.length - 1; i >= 0; i--) {
        listeners[i].controllersProbed();
      }
    }
    final int result = getButtons();
    buttonsAND = -1;
    buttonsOR = 0;
    return result;
  }
  
  public void spriteZeroHit(final int scanline, final int scanlineCycle) {
    final SpriteZeroListener[] listeners = spriteZeroListeners;
    if (listeners != null) {
      for(int i = listeners.length - 1; i >= 0; i--) {
        listeners[i].spriteZeroHit(scanline, scanlineCycle);        
      }
    }
  }
  
  public void machineStopped() {
    stopped = true;      
    final DeactivateListener[] listeners = deactivateListeners;
    if (listeners != null) {
      for(int i = listeners.length - 1; i >= 0; i--) {
        listeners[i].apiDisabled();
      }
    }
  }
  
  public void setPalettePPU(final PalettePPU palettePPU) {
    final int[] map = palettePPU.getMap();
    updateColor(map);
    for(final Sprite sprite : sprites.values()) {
      updateSpriteImage(sprite, map);
    }
  }
  
  private void updateSpriteImage(final Sprite sprite, final int[] map) {    
    final int[] pixels = sprite.pixels;
    final int[] data = ((DataBufferInt)sprite.image.getRaster()
        .getDataBuffer()).getData();
    for(int i = pixels.length - 1; i >= 0; i--) {
      if (pixels[i] < 0) {
        data[i] = 0;
      } else {
        data[i] = 0xFF000000 | (0x1C0 & pixels[i]) | map[0x03F & pixels[i]];
      }
    }
  } 
  
  private void updateColor(final int[] map) {
    g.setColor(new Color(0xFF000000 | (0x1C0 & requestColor) 
        | map[0x03F & requestColor]));
  }
  
  private int getZapperShift() {
    final Mapper mapper = _mapper;
    if (mapper == null) {
      return 8;
    } else {
      final ZapperMapper zapper = (ZapperMapper)mapper.getDeviceMapper(
          InputDevices.Zapper);
      return zapper == null ? 8 : zapper.getShift();
    }
  }
  
  private int getButtons() {
    return (buttons & buttonsAND) | buttonsOR;
  }

  @Override
  @ApiMethod(value = Activate, autogenerated = false)
  public void addActivateListener(final ActivateListener listener) {
    if (listener != null) {
      activateListeners = addElement(ActivateListener.class, activateListeners,
          listener);
    }
  }

  @Override
  @ApiMethod(value = 2, autogenerated = false)
  public void removeActivateListener(final ActivateListener listener) {
    activateListeners = removeAllElements(ActivateListener.class, 
        activateListeners, listener);
  }

  @Override
  @ApiMethod(value = Deactivate, autogenerated = false)
  public void addDeactivateListener(final DeactivateListener listener) {
    if (listener != null) {
      deactivateListeners = addElement(DeactivateListener.class, 
          deactivateListeners, listener);
    }
  }

  @Override
  @ApiMethod(value = 4, autogenerated = false)
  public void removeDeactivateListener(final DeactivateListener listener) {
    deactivateListeners = removeAllElements(DeactivateListener.class, 
        deactivateListeners, listener);
  }

  @Override
  @ApiMethod(value = Stop, autogenerated = false)
  public void addStopListener(final StopListener listener) {
    if (listener != null) {
      stopListeners = addElement(StopListener.class, stopListeners, listener);
    }
  }

  @Override
  @ApiMethod(value = 6, autogenerated = false)
  public void removeStopListener(final StopListener listener) {
    stopListeners = removeAllElements(StopListener.class, stopListeners, 
        listener);
  }
  
  @Override
  @ApiMethod(value = 7, autogenerated = false)
  public void addAccessPointListener(final AccessPointListener listener, 
      final int accessPointType, final int address) {    
    addAccessPointListener(listener, accessPointType, address, -1, -1);
  }

  @Override
  @ApiMethod(value = 8, autogenerated = false)
  public void addAccessPointListener(final AccessPointListener listener, 
      final int accessPointType, final int minAddress, final int maxAddress) {    
    addAccessPointListener(listener, accessPointType, minAddress, maxAddress, 
        -1);
  }

  @Override
  @ApiMethod(value = Access, autogenerated = false)
  public void addAccessPointListener(final AccessPointListener listener, 
      final int accessPointType, final int minAddress, final int maxAddress, 
          final int bank) {
    if (listener != null) {
      accessPoints = addElement(AccessPoint.class, accessPoints, 
          new AccessPoint(listener, accessPointType, minAddress, maxAddress, 
              bank));
      final CPU cpu = _cpu;
      if (cpu != null) {
        cpu.setAccessPoints(accessPoints);
      }
    }
  }

  @Override
  @ApiMethod(value = 10, autogenerated = false)
  public void removeAccessPointListener(final AccessPointListener listener) {
    accessPoints = removeAll(AccessPoint.class, accessPoints, 
        a -> a.listener == listener);
    final CPU cpu = _cpu;
    if (cpu != null) {
      cpu.setAccessPoints(accessPoints);
    }
  }

  @Override
  @ApiMethod(value = Controllers, autogenerated = false)
  public void addControllersListener(final ControllersListener listener) {
    if (listener != null) {
      controllersListeners = addElement(ControllersListener.class, 
          controllersListeners, listener);
    }
  }

  @Override
  @ApiMethod(value = 12, autogenerated = false)
  public void removeControllersListener(final ControllersListener listener) {
    controllersListeners = removeAllElements(ControllersListener.class, 
        controllersListeners, listener);
  }

  @Override
  @ApiMethod(value = Frame, autogenerated = false)
  public void addFrameListener(final FrameListener listener) {
    if (listener != null) {
      frameListeners = addElement(FrameListener.class, frameListeners, 
          listener);
    }
  }

  @Override
  @ApiMethod(value = 14, autogenerated = false)
  public void removeFrameListener(final FrameListener listener) {
    frameListeners = removeAllElements(FrameListener.class, frameListeners, 
        listener);
  }

  @Override
  @ApiMethod(value = Scanline, autogenerated = false)
  public void addScanlineListener(final ScanlineListener listener, 
      final int scanline) {   
    if (listener != null || scanline < -1 
        || scanline >= PAL.getScanlineCount() - 1) {
      scanlinePoints = addElement(ScanlinePoint.class, scanlinePoints,
          new ScanlinePoint(listener, scanline));
    }
  }

  @Override
  @ApiMethod(value = 16, autogenerated = false)
  public void removeScanlineListener(final ScanlineListener listener) {
    scanlinePoints = removeAll(ScanlinePoint.class, scanlinePoints, 
        s -> s.listener == listener);
  }

  @Override
  @ApiMethod(value = ScanlineCycle, autogenerated = false)
  public void addScanlineCycleListener(final ScanlineCycleListener listener,
      final int scanline, final int scanlineCycle) {
    
    if (listener == null || scanline < -1 
        || scanline >= PAL.getScanlineCount() - 1 || scanlineCycle < 0 
            || scanlineCycle > 340) {
      return;
    }
    
    if (scanlineCyclePoints == null) {
      scanlineCyclePoints = new ScanlineCyclePoint[PAL.getScanlineCount()][][];
    }
    
    final int sl = scanline + 1;
    if (scanlineCyclePoints[sl] == null) {
      scanlineCyclePoints[sl] = new ScanlineCyclePoint[341][];
    }
    
    scanlineCyclePoints[sl][scanlineCycle] = addElement(
        ScanlineCyclePoint.class, scanlineCyclePoints[sl][scanlineCycle],
            new ScanlineCyclePoint(listener, scanline, scanlineCycle));
  }

  @Override
  @ApiMethod(value = 18, autogenerated = false)
  public void removeScanlineCycleListener(
      final ScanlineCycleListener listener) {
    
    if (scanlineCyclePoints == null) {
      return;
    }
    
    boolean allRowsEmpty = true;
    for(int i = PAL.getScanlineCount() - 1; i >= 0; i--) {
      if (scanlineCyclePoints[i] != null) {
        final ScanlineCyclePoint[][] row = scanlineCyclePoints[i];
        boolean rowEmpty = true;
        for(int j = 340; j >= 0; j--) {
          if (row[j] != null) {
            row[j] = removeAll(ScanlineCyclePoint.class, row[j], 
                s -> s.listener == listener);
          }
          if (row[j] != null) {
            rowEmpty = false;
          }
        }
        if (rowEmpty) {
          scanlineCyclePoints[i] = null;
        }
      }
      if (scanlineCyclePoints[i] != null) {
        allRowsEmpty = false;
      }
    }
    if (allRowsEmpty) {
      scanlineCyclePoints = null;
    }
  }
  
  @Override
  @ApiMethod(value = SpriteZero, autogenerated = false)
  public void addSpriteZeroListener(final SpriteZeroListener listener) {
    if (listener != null) {
      spriteZeroListeners = addElement(SpriteZeroListener.class, 
          spriteZeroListeners, listener);
    }
  }

  @Override
  @ApiMethod(value = 20, autogenerated = false)
  public void removeSpriteZeroListener(final SpriteZeroListener listener) {
    spriteZeroListeners = removeAllElements(SpriteZeroListener.class, 
        spriteZeroListeners, listener);
  }  

  @Override
  @ApiMethod(value = Status, autogenerated = false)
  public void addStatusListener(final StatusListener listener) {
    if (listener != null) {
      statusListeners = addElement(StatusListener.class, statusListeners, 
          listener);
    }
  }

  @Override
  @ApiMethod(value = 22, autogenerated = false)
  public void removeStatusListener(final StatusListener listener) {
    statusListeners = removeAllElements(StatusListener.class, statusListeners, 
        listener);
  }
  
  @Override
  @ApiMethod(23)
  public void setPaused(final boolean paused) {
    App.setStepPause(paused);
  }

  @Override
  @ApiMethod(24)
  public boolean isPaused() {
    final MachineRunner runner = App.getMachineRunner();
    if (runner == null) {
      return false;
    }
    return runner.isPaused();
  }

  @Override
  @ApiMethod(25)
  public int getFrameCount() {
    final PPU ppu = _ppu;
    if (ppu == null) {
      return -1;
    }
    return ppu.getFrameCounter();
  }

  @Override
  @ApiMethod(26)
  public int getA() {
    final CPU cpu = _cpu;
    if (cpu == null) {
      return -1;
    }
    return cpu.getA();
  }

  @Override
  @ApiMethod(27)
  public void setA(final int A) {
    final CPU cpu = _cpu;
    if (cpu != null) {
      cpu.setA(A);
    }
  }

  @Override
  @ApiMethod(28)
  public int getS() {
    final CPU cpu = _cpu;
    if (cpu == null) {
      return -1;
    }
    return cpu.getS();
  }

  @Override
  @ApiMethod(29)
  public void setS(final int S) {
    final CPU cpu = _cpu;
    if (cpu != null) {
      cpu.setS(S);
    }
  }

  @Override
  @ApiMethod(30)
  public int getPC() {
    final CPU cpu = _cpu;
    if (cpu == null) {
      return -1;
    }
    return cpu.getPC();
  }

  @Override
  @ApiMethod(31)
  public void setPC(int PC) {
    final CPU cpu = _cpu;
    if (cpu != null) {
      cpu.setPC(PC);
    }
  }

  @Override
  @ApiMethod(32)
  public int getX() {
    final CPU cpu = _cpu;
    if (cpu == null) {
      return -1;
    }
    return cpu.getX();
  }

  @Override
  @ApiMethod(33)
  public void setX(final int X) {
    final CPU cpu = _cpu;
    if (cpu != null) {
      cpu.setX(X);
    }
  }

  @Override
  @ApiMethod(34)
  public int getY() {
    final CPU cpu = _cpu;
    if (cpu == null) {
      return -1;
    }
    return cpu.getY();
  }

  @Override
  @ApiMethod(35)
  public void setY(final int Y) {
    final CPU cpu = _cpu;
    if (cpu != null) {
      cpu.setY(Y);
    }
  }

  @Override
  @ApiMethod(36)
  public int getP() {
    final CPU cpu = _cpu;
    if (cpu == null) {
      return -1;
    }
    return cpu.getP();
  }

  @Override
  @ApiMethod(37)
  public void setP(final int P) {
    final CPU cpu = _cpu;
    if (cpu != null) {
      cpu.setP(P);
    }
  }

  @Override
  @ApiMethod(38)
  public boolean isN() {
    final CPU cpu = _cpu;
    if (cpu == null) {
      return false;
    }
    return toBitBool(cpu.getN());
  }

  @Override
  @ApiMethod(39)
  public void setN(final boolean N) {
    final CPU cpu = _cpu;
    if (cpu != null) {
      cpu.setN(toBit(N));
    }
  }

  @Override
  @ApiMethod(40)
  public boolean isV() {
    final CPU cpu = _cpu;
    if (cpu == null) {
      return false;
    }
    return toBitBool(cpu.getV());
  }

  @Override
  @ApiMethod(41)
  public void setV(final boolean V) {
    final CPU cpu = _cpu;
    if (cpu != null) {
      cpu.setV(toBit(V));
    }
  }

  @Override
  @ApiMethod(42)
  public boolean isD() {
    final CPU cpu = _cpu;
    if (cpu == null) {
      return false;
    }
    return toBitBool(cpu.getD());
  }

  @Override
  @ApiMethod(43)
  public void setD(final boolean D) {
    final CPU cpu = _cpu;
    if (cpu != null) {
      cpu.setD(toBit(D));
    }
  }

  @Override
  @ApiMethod(44)
  public boolean isI() {
    final CPU cpu = _cpu;
    if (cpu == null) {
      return false;
    }
    return toBitBool(cpu.getI());
  }

  @Override
  @ApiMethod(45)
  public void setI(final boolean I) {
    final CPU cpu = _cpu;
    if (cpu != null) {
      cpu.setI(toBit(I));
    }
  }

  @Override
  @ApiMethod(46)
  public boolean isZ() {
    final CPU cpu = _cpu;
    if (cpu == null) {
      return false;
    }
    return toBitBool(cpu.getZ());
  }

  @Override
  @ApiMethod(47)
  public void setZ(final boolean Z) {
    final CPU cpu = _cpu;
    if (cpu != null) {
      cpu.setZ(toBit(Z));
    }
  }

  @Override
  @ApiMethod(48)
  public boolean isC() {
    final CPU cpu = _cpu;
    if (cpu == null) {
      return false;
    }
    return toBitBool(cpu.getC());
  }

  @Override
  @ApiMethod(49)
  public void setC(final boolean C) {
    final CPU cpu = _cpu;
    if (cpu != null) {
      cpu.setC(toBit(C));
    }
  }

  @Override
  @ApiMethod(50)
  public int getPPUv() {
    final PPU ppu = _ppu;
    if (ppu == null) {
      return -1;
    }
    return ppu.getV();
  }

  @Override
  @ApiMethod(51)
  public void setPPUv(final int v) {
    final PPU ppu = _ppu;
    if (ppu != null) {
      ppu.setV(v);
    }
  }  

  @Override
  @ApiMethod(52)
  public int getPPUt() {
    final PPU ppu = _ppu;
    if (ppu == null) {
      return -1;
    }
    return ppu.getT();
  }

  @Override
  @ApiMethod(53)
  public void setPPUt(final int t) {
    final PPU ppu = _ppu;
    if (ppu != null) {
      ppu.setT(t);
    }
  }

  @Override
  @ApiMethod(54)
  public int getPPUx() {
    final PPU ppu = _ppu;
    if (ppu == null) {
      return -1;
    }
    return ppu.getX();
  }

  @Override
  @ApiMethod(55)
  public void setPPUx(final int x) {
    final PPU ppu = _ppu;
    if (ppu != null) {
      ppu.setX(x);
    }
  }

  @Override
  @ApiMethod(56)
  public boolean isPPUw() {
    final PPU ppu = _ppu;
    if (ppu == null) {
      return false;
    }
    return ppu.isW();
  }

  @Override
  @ApiMethod(57)
  public void setPPUw(final boolean w) {
    final PPU ppu = _ppu;
    if (ppu != null) {
      ppu.setW(w);
    }
  }

  @Override
  @ApiMethod(58)
  public int getCameraX() {
    final PPU ppu = _ppu;
    if (ppu == null) {
      return -1;
    }
    return ppu.getScrollX();
  }

  @Override
  @ApiMethod(59)
  public void setCameraX(final int scrollX) {
    final PPU ppu = _ppu;
    if (ppu != null) {
      ppu.setScrollX(scrollX);
    }
  }

  @Override
  @ApiMethod(60)
  public int getCameraY() {
    final PPU ppu = _ppu;
    if (ppu == null) {
      return -1;
    }
    return ppu.getScrollY();
  }

  @Override
  @ApiMethod(61)
  public void setCameraY(final int scrollY) {
    final PPU ppu = _ppu;
    if (ppu != null) {
      ppu.setScrollY(scrollY);
    }
  }

  @Override
  @ApiMethod(62)
  public int getScanline() {
    final PPU ppu = _ppu;
    if (ppu == null) {
      return -1;
    }
    return ppu.getScanline();
  }

  @Override
  @ApiMethod(63)
  public int getDot() {
    final PPU ppu = _ppu;
    if (ppu == null) {
      return -1;
    }
    return ppu.getScanlineCycle();
  }

  @Override
  @ApiMethod(64)
  public boolean isSpriteZeroHit() {
    final PPU ppu = _ppu;
    if (ppu == null) {
      return false;
    }
    return ppu.isSprite0Hit();
  }

  @Override
  @ApiMethod(65)
  public void setSpriteZeroHit(final boolean sprite0Hit) {
    final PPU ppu = _ppu;
    if (ppu != null) {
      ppu.setSprite0Hit(sprite0Hit);
    }
  }

  @Override
  @ApiMethod(66)
  public int getScanlineCount() {
    final PPU ppu = _ppu;
    if (ppu == null) {
      return -1;
    }
    return ppu.getScanlineCount();
  }

  @Override
  @ApiMethod(67)
  public void requestInterrupt() {
    final CPU cpu = _cpu;
    if (cpu != null) {
      cpu.setMapperIrq(true);
    }
  }

  @Override
  @ApiMethod(68)
  public void acknowledgeInterrupt() {
    final CPU cpu = _cpu;
    if (cpu != null) {
      cpu.setMapperIrq(false);
    }
  }

  @Override
  @ApiMethod(69)
  public int peekCPU(final int address) {
    final Mapper mapper = _mapper;
    if (mapper == null) {
      return -1;
    }
    return mapper.peekCpuMemory(address);
  }

  @Override
  @ApiMethod(70)
  public int readCPU(final int address) {
    final Mapper mapper = _mapper;
    if (mapper == null) {
      return -1;
    }
    return mapper.readCpuMemory(address);
  }

  @Override
  @ApiMethod(71)
  public void writeCPU(final int address, final int value) {
    final Mapper mapper = _mapper;
    if (mapper != null) {
      mapper.writeCpuMemory(address, value);
    }    
  }

  @Override
  @ApiMethod(72)
  public int peekCPU16(final int address) {
    final Mapper mapper = _mapper;
    if (mapper == null) {
      return -1;
    }
    final int b0 = mapper.peekCpuMemory(address);
    final int b1 = mapper.peekCpuMemory(address + 1);
    return (b1 << 8) | b0;
  }

  @Override
  @ApiMethod(73)
  public int readCPU16(final int address) {
    final Mapper mapper = _mapper;
    if (mapper == null) {
      return -1;
    }
    final int b0 = mapper.readCpuMemory(address);
    final int b1 = mapper.readCpuMemory(address + 1);
    return (b1 << 8) | b0;
  }

  @Override
  @ApiMethod(74)
  public void writeCPU16(final int address, final int value) {
    final Mapper mapper = _mapper;
    if (mapper != null) {
      mapper.writeCpuMemory(address, value & 0xFF);
      mapper.writeCpuMemory(address + 1, (value >> 8) & 0xFF);
    }
  }

  @Override
  @ApiMethod(75)
  public int peekCPU32(final int address) {
    final Mapper mapper = _mapper;
    if (mapper == null) {
      return -1;
    }
    final int b0 = mapper.peekCpuMemory(address);
    final int b1 = mapper.peekCpuMemory(address + 1);
    final int b2 = mapper.peekCpuMemory(address + 2);
    final int b3 = mapper.peekCpuMemory(address + 3);
    return (b3 << 24) | (b2 << 16) | (b1 << 8) | b0;
  }

  @Override
  @ApiMethod(76)
  public int readCPU32(final int address) {
    final Mapper mapper = _mapper;
    if (mapper == null) {
      return -1;
    }
    final int b0 = mapper.readCpuMemory(address);
    final int b1 = mapper.readCpuMemory(address + 1);
    final int b2 = mapper.readCpuMemory(address + 2);
    final int b3 = mapper.readCpuMemory(address + 3);
    return (b3 << 24) | (b2 << 16) | (b1 << 8) | b0;
  }

  @Override
  @ApiMethod(77)
  public void writeCPU32(final int address, final int value) {
    final Mapper mapper = _mapper;
    if (mapper != null) {
      mapper.writeCpuMemory(address, value & 0xFF);
      mapper.writeCpuMemory(address + 1, (value >> 8) & 0xFF);
      mapper.writeCpuMemory(address + 2, (value >> 16) & 0xFF);
      mapper.writeCpuMemory(address + 3, (value >> 24) & 0xFF);
    }
  }

  @Override
  @ApiMethod(78)
  public int readPPU(final int address) {
    final Mapper mapper = _mapper;
    if (mapper == null) {
      return -1;
    }
    return mapper.readVRAM(mapper.maskVRAMAddress(address));
  }

  @Override
  @ApiMethod(79)
  public void writePPU(final int address, final int value) {
    final Mapper mapper = _mapper;
    if (mapper != null) {
      mapper.writeVRAM(mapper.maskVRAMAddress(address), value);
    }
  }

  @Override
  @ApiMethod(80)
  public int readPaletteRAM(final int address) {
    final PPU ppu = _ppu;
    if (ppu == null) {
      return -1;
    }
    return ppu.getPaletteRamValue(address);
  }

  @Override
  @ApiMethod(81)
  public void writePaletteRAM(final int address, final int value) {
    final PPU ppu = _ppu;
    if (ppu != null) {
      ppu.setPaletteRamValue(address, value);
    }
  }

  @Override
  @ApiMethod(82)
  public int readOAM(final int address) {
    final PPU ppu = _ppu;
    if (ppu == null) {
      return -1;
    }
    return ppu.getOAM()[address & 0xFF];
  }

  @Override
  @ApiMethod(83)
  public void writeOAM(final int address, final int value) {
    final PPU ppu = _ppu;
    if (ppu != null) {
      ppu.getOAM()[address & 0xFF] = value;
    }
  }

  @Override
  @ApiMethod(84)
  public boolean readGamepad(final int gamepad, final int button) {
    return getBitBool(getButtons(), ((gamepad & 3) << 3) | (button & 7));
  }

  @Override
  @ApiMethod(85)
  public void writeGamepad(final int gamepad, final int button, 
      final boolean value) {
    final int bit = ((gamepad & 3) << 3) | (button & 7);
    buttonsAND = resetBit(buttonsAND, bit);
    buttonsOR = setBit(buttonsOR, bit, value);
  }

  @Override
  @ApiMethod(86)
  public boolean isZapperTrigger() {
    return getBitBool(getButtons(), getZapperShift() + 2);
  }

  @Override
  @ApiMethod(87)
  public void setZapperTrigger(final boolean zapperTrigger) {
    final int bit = getZapperShift() + 2;
    buttonsAND = resetBit(buttonsAND, bit);
    buttonsOR = setBit(buttonsOR, bit, zapperTrigger);
  }

  @Override
  @ApiMethod(88)
  public int getZapperX() {
    final int bs = getButtons();
    if ((bs & 0xFFFF0000) == 0xFFFF0000) {
      return -1;
    }
    return (bs >> 16) & 0xFF;
  }

  @Override
  @ApiMethod(89)
  public void setZapperX(final int x) {    
    if (x < 0 || x > 255) {      
      buttonsAND &= 0x0000FFFF;
      buttonsOR = 0xFFFF0000 | (buttonsOR & 0x0000FFFF);
    } else {
      buttonsAND &= 0xFF00FFFF;
      buttonsOR = (x << 16) | (buttonsOR & 0xFF00FFFF);
    }
  }

  @Override
  @ApiMethod(90)
  public int getZapperY() {
    final int bs = getButtons();
    if ((bs & 0xFFFF0000) == 0xFFFF0000) {
      return -1;
    }
    return (bs >> 24) & 0xFF;
  }

  @Override
  @ApiMethod(91)
  public void setZapperY(final int y) {
    if (y < 0 || y > 239) {
      buttonsAND &= 0x0000FFFF;
      buttonsOR = 0xFFFF0000 | (buttonsOR & 0x0000FFFF);
    } else {
      buttonsAND &= 0x00FFFFFF;
      buttonsOR = (y << 24) | (buttonsOR & 0x00FFFFFF);
    }
  }

  @Override
  @ApiMethod(92)
  public void setColor(final int color) {
    requestColor = color;    
    updateColor(PaletteUtil.getPalettePPU().getMap());
  }

  @Override
  @ApiMethod(93)
  public int getColor() {
    return requestColor;
  }

  @Override
  @ApiMethod(94)
  public void setClip(final int x, final int y, final int width, 
      final int height) {
    g.setClip(x, y, width, height);
  }

  @Override
  @ApiMethod(95)
  public void clipRect(final int x, final int y, final int width, 
      final int height) {
    g.clipRect(x, y, width, height);
  }

  @Override
  @ApiMethod(96)
  public void resetClip() {
    g.setClip(null);
  }

  @Override
  @ApiMethod(97)
  public void copyArea(final int x, final int y, final int width, 
      final int height, final int dx, final int dy) {
    pixelsModified = true;
    g.copyArea(x, y, width, height, dx, dy);
  }

  @Override
  @ApiMethod(98)
  public void drawLine(final int x1, final int y1, final int x2, final int y2) {
    pixelsModified = true;
    g.drawLine(x1, y1, x2, y2);
  }

  @Override
  @ApiMethod(99)
  public void drawOval(final int x, final int y, final int width, 
      final int height) {
    pixelsModified = true;
    g.drawOval(x, y, width, height);
  }

  @Override
  @ApiMethod(100)
  public void drawPolygon(final int[] xPoints, final int[] yPoints, 
      final int nPoints) {
    pixelsModified = true;
    g.drawPolygon(xPoints, yPoints, nPoints);
  }

  @Override
  @ApiMethod(101)
  public void drawPolyline(final int[] xPoints, final int[] yPoints, 
      final int nPoints) {
    pixelsModified = true;
    g.drawPolyline(xPoints, yPoints, nPoints);
  }

  @Override
  @ApiMethod(102)
  public void drawRect(final int x, final int y, final int width, 
      final int height) {
    pixelsModified = true;
    g.drawRect(x, y, width, height);
  }

  @Override
  @ApiMethod(103)
  public void drawRoundRect(final int x, final int y, final int width, 
      final int height, final int arcWidth, final int arcHeight) {
    pixelsModified = true;
    g.drawRoundRect(x, y, width, height, arcWidth, arcHeight);
  }

  @Override
  @ApiMethod(104)
  public void draw3DRect(final int x, final int y, final int width, 
      final int height, final boolean raised) {
    pixelsModified = true;
    g.draw3DRect(x, y, width, height, raised);
  }

  @Override
  @ApiMethod(105)
  public void drawArc(final int x, final int y, final int width, 
      final int height, final int startAngle, final int arcAngle) {
    pixelsModified = true;
    g.drawArc(x, y, width, height, startAngle, arcAngle);
  }

  @Override
  @ApiMethod(106)
  public void fill3DRect(final int x, final int y, final int width, 
      final int height, final boolean raised) {
    pixelsModified = true;
    g.fill3DRect(x, y, width, height, raised);
  }

  @Override
  @ApiMethod(107)
  public void fillArc(final int x, final int y, final int width, 
      final int height, final int startAngle, final int arcAngle) {
    pixelsModified = true;
    g.fillArc(x, y, width, height, startAngle, arcAngle);
  }

  @Override
  @ApiMethod(108)
  public void fillOval(final int x, final int y, final int width, 
      final int height) {
    pixelsModified = true;
    g.fillOval(x, y, width, height);
  }

  @Override
  @ApiMethod(109)
  public void fillPolygon(final int[] xPoints, final int[] yPoints, 
      final int nPoints) {
    pixelsModified = true;
    g.fillPolygon(xPoints, yPoints, nPoints);
  }

  @Override
  @ApiMethod(110)
  public void fillRect(final int x, final int y, final int width, 
      final int height) {
    pixelsModified = true;
    g.fillRect(x, y, width, height);
  }

  @Override
  @ApiMethod(111)
  public void fillRoundRect(final int x, final int y, final int width, 
      final int height, final int arcWidth, final int arcHeight) {
    pixelsModified = true;
    g.fillRoundRect(x, y, width, height, arcWidth, arcHeight);
  }
  
  @Override
  @ApiMethod(112)
  public void drawChar(final char c, final int x, final int y) {
    pixelsModified = true;
    FontUtil.setColor(requestColor);
    FontUtil.drawChar(g, c, x, y);
  }

  @Override
  @ApiMethod(113)
  public void drawChars(final char[] data, final int offset, final int length, 
      final int x, final int y, final boolean monospaced) {
    pixelsModified = true;
    FontUtil.setColor(requestColor);
    FontUtil.drawChars(g, data, offset, length, x, y, monospaced);
  }

  @Override
  @ApiMethod(114)
  public void drawString(final String str, final int x, final int y,
      final boolean monospaced) {
    pixelsModified = true;
    FontUtil.setColor(requestColor);
    FontUtil.drawString(g, str, x, y, monospaced);
  }

  @Override
  @ApiMethod(115)
  public void createSprite(final int id, final int width, final int height,
      int[] pixels) {
    
    if (pixels == null || width <= 0 || height <= 0) {
      return;
    }

    final int length = width * height;
    if (pixels.length != length) {
      final int[] ps = new int[length];
      System.arraycopy(pixels, 0, ps, 0, min(pixels.length, length));
      pixels = ps;
    }
    
    final Sprite sprite = new Sprite();
    sprite.pixels = pixels;
    sprite.image = new BufferedImage(width, height, TYPE_INT_ARGB);
    updateSpriteImage(sprite, PaletteUtil.getPalettePPU().getMap());
    sprites.put(id, sprite);
  }
  
  @Override
  @ApiMethod(116)
  public void drawSprite(final int id, final int x, final int y) {
    
    final BufferedImage image = sprites.get(id).image;
    if (image == null) {
      return;
    }
    
    pixelsModified = true;
    g.drawImage(image, x, y, null);
  }  

  @Override
  @ApiMethod(117)
  public void setPixel(final int x, final int y, final int color) {
    
    if (x < 0 || y < 0 || x >= IMAGE_WIDTH || y >= IMAGE_HEIGHT || color < 0) {
      return;
    }
    
    pixelsModified = true;
    pixels[(y << 8) | x] = 0x1FF & color;
  }

  @Override
  @ApiMethod(118)
  public int getPixel(final int x, final int y) {
    
    if (x < 0 || y < 0 || x >= IMAGE_WIDTH || y >= IMAGE_HEIGHT) {
      return -1;
    }
    
    final int index = (y << 8) | x;
    if (pixelsModified) {      
      final int value = pixels[index];
      if (value != 0) {
        return value;
      }
    }
    
    final int[] s = screen;
    if (s == null) {
      return -1;
    }
    
    return 0x1FF & screen[index];
  }

  @Override
  @ApiMethod(value = 119, autogenerated = false)
  public void getPixels(final int[] pixels) {
    
    if (pixels == null) {
      return;
    }
    
    final int[] s = screen;
    if (s == null) {
      return;
    }
    
    System.arraycopy(s, 0, pixels, 0, min(s.length, pixels.length));
  }
  
  public int[] getScreen() {
    return screen;
  }
  
  @Override
  @ApiMethod(120)
  public void powerCycle() {    
    App.powerCycle();
  }
  
  @Override
  @ApiMethod(121)
  public void reset() {
    App.reset();
  }  
  
  @Override
  @ApiMethod(122)
  public void deleteSprite(final int id) {
    sprites.remove(id);
  }

  @Override
  @ApiMethod(123)
  public void setSpeed(final int percent) {
    App.setSpeed(percent);
  }

  @Override
  @ApiMethod(124)
  public void stepToNextFrame() {
    App.step(PauseStepType.Frame);
  }

  @Override
  @ApiMethod(125)
  public void showMessage(final String message) {
    App.showMessage(message);
  }

  @Override
  @ApiMethod(126)
  public String getWorkingDirectory() {
    return FileUtil.getWorkingDirectory();
  }

  @Override
  @ApiMethod(127)
  public String getContentDirectory() {
    return AppPrefs.getInstance().getPaths().getContentDirectory();
  }

  @Override
  @ApiMethod(128)
  public void open(final String fileName) {
    App.getImageFrame().open(new FilePath(fileName));
  }

  @Override
  @ApiMethod(129)
  public void openArchiveEntry(final String archiveFileName, 
      final String entryFileName) {
    App.getImageFrame().open(new FilePath(entryFileName, archiveFileName));
  }
  
  @Override
  @ApiMethod(130)
  public String[] getArchiveEntries(final String archiveFileName) {
    List<String> entries = null;
    try {
      entries = ArchiveEntry.toNames(FileUtil.getArchiveEntries(
          archiveFileName));
    } catch(final Throwable t) {      
    }
    if (entries == null) {
      return new String[0];
    } else {
      return entries.toArray(new String[entries.size()]);
    }    
  }

  @Override
  @ApiMethod(131)
  public String getDefaultArchiveEntry(final String archiveFileName) {
    List<String> entries = null;
    try {
      entries = ArchiveEntry.toNames(FileUtil.getArchiveEntries(
          archiveFileName));
    } catch(Throwable t) {      
    }
    if (isBlank(entries)) {
      return "";
    }
    final int index = FileUtil.getDefaultArchiveEntry(archiveFileName, entries);
    if (index < 0) {
      return "";
    } else {
      return entries.get(index);
    }
  }

  @Override
  @ApiMethod(132)
  public void openDefaultArchiveEntry(final String archiveFileName) {
    final String entryFileName = getDefaultArchiveEntry(archiveFileName);
    if (!isBlank(entryFileName)) {
      openArchiveEntry(archiveFileName, entryFileName);
    }
  }

  @Override
  @ApiMethod(133)
  public void close() {
    App.close();
  }

  @Override
  @ApiMethod(134)
  public void saveState(final String stateFileName) {
    EventQueue.invokeLater(() -> App.getImageFrame().saveState(new File(
        stateFileName)));
  }

  @Override
  @ApiMethod(135)
  public void loadState(final String stateFileName) {
    EventQueue.invokeLater(() -> App.getImageFrame().loadState(new File(
        stateFileName)));
  }

  @Override
  @ApiMethod(136)
  public void quickSaveState(final int slot) {
    if (slot >= 0 && slot <= 9) {
      EventQueue.invokeLater(() -> App.getImageFrame().quickSaveState(slot));
    }
  }

  @Override
  @ApiMethod(137)
  public void quickLoadState(final int slot) {
    if (slot >= 0 && slot <= 9) {
      EventQueue.invokeLater(() -> App.getImageFrame().quickLoadState(slot));
    }
  }

  @Override
  @ApiMethod(138)
  public void setTVSystem(final String tvSystem) {
    switch(tvSystem.toUpperCase(Locale.ENGLISH)) {
      case "NTSC":
        InputUtil.addOtherInput(new SetTVSystem(NTSC));
        break;
      case "PAL":
        InputUtil.addOtherInput(new SetTVSystem(PAL));
        break;
      case "DENDY":
        InputUtil.addOtherInput(new SetTVSystem(Dendy));
        break;
    }    
  }

  @Override
  @ApiMethod(139)
  public String getTVSystem() {
    final Mapper mapper = _mapper;
    if (mapper != null) {
      switch(mapper.getTVSystem()) {
        case NTSC:
          return "NTSC";
        case PAL:
          return "PAL";
        case Dendy:
          return "Dendy";
      }
    }
    return "NTSC";
  }

  @Override
  @ApiMethod(140)
  public int getDiskSides() {
    final Mapper mapper = _mapper;
    if (mapper != null) {
      return mapper.getDiskSideCount();
    } else {
      return 0;
    }
  }

  @Override
  @ApiMethod(141)
  public void insertDisk(int disk, int side) {
    if (side >= 0 && side <= 1) {
      final int index = (disk << 1) | side;
      if (index < getDiskSides()) {
        InputUtil.addOtherInput(new SetDiskSide(index));
      }
    }
  }

  @Override
  @ApiMethod(142)
  public void flipDiskSide() {
    InputUtil.addOtherInput(new FlipDiskSide());
  }

  @Override
  @ApiMethod(143)
  public void ejectDisk() {
    InputUtil.addOtherInput(new EjectDisk());
  }

  @Override
  @ApiMethod(144)
  public void insertCoin() {
    InputUtil.addOtherInput(new InsertCoin(VsSystem.Main, CoinSlot.Left));
  }

  @Override
  @ApiMethod(145)
  public void pressServiceButton() {
    InputUtil.addOtherInput(new PressServiceButton(VsSystem.Main));
  }

  @Override
  @ApiMethod(146)
  public void screamIntoMicrophone() {
    InputUtil.addOtherInput(new ScreamIntoMicrophone());
  }

  @Override
  @ApiMethod(147)
  public void glitch() {
    InputUtil.addOtherInput(new Glitch());
  }

  @Override
  @ApiMethod(148)
  public String getFileInfo() {
    final String fileInfo = App.getImageFrame().getFileInfo();
    if (fileInfo == null) {
      return "";
    } else {
      return fileInfo;
    }
  }

  @Override
  @ApiMethod(149)
  public void setFullscreenMode(final boolean fullscreenMode) {
    EventQueue.invokeLater(() -> App.getImageFrame()
        .setFullscreenMode(fullscreenMode));
  }

  @Override
  @ApiMethod(150)
  public void saveScreenshot() {
    App.getImageFrame().getImagePane().requestScreenshot();
  }

  @Override
  @ApiMethod(151)
  public void addCheat(final int address, final int value, final int compare, 
      final String description, final boolean enabled) {    
    final Cheat cheat = new Cheat(address, value, compare);
    cheat.setDescription(description);
    cheat.setEnabled(enabled);
    GameCheats.addCheat(cheat, true);
    GameCheats.save();
    GameCheats.updateMachine();
  }

  @Override
  @ApiMethod(152)
  public void removeCheat(final int address, final int value, 
      final int compare) {
    if (GameCheats.removeCheat(new Cheat(address, value, compare))) {
      GameCheats.save();
      GameCheats.updateMachine();
    }
  }

  @Override
  @ApiMethod(153)
  public void addGameGenie(final String gameGenieCode, final String description,
      final boolean enabled) {    
    final Cheat cheat = GameGenie.convert(gameGenieCode);
    if (cheat != null) {
      cheat.setDescription(description);
      cheat.setEnabled(enabled);
      GameCheats.addCheat(cheat, true);
      GameCheats.save();
      GameCheats.updateMachine();
    }
  }

  @Override
  @ApiMethod(154)
  public void removeGameGenie(final String gameGenieCode) {
    final Cheat cheat = GameGenie.convert(gameGenieCode);
    if (cheat != null && GameCheats.removeCheat(cheat)) {
      GameCheats.save();
      GameCheats.updateMachine();
    }
  }

  @Override
  @ApiMethod(155)
  public void addProActionRocky(final String proActionRockyCode, 
      final String description, final boolean enabled) {    
    final Cheat cheat = ProActionRocky.convert(proActionRockyCode);
    if (cheat != null) {
      cheat.setDescription(description);
      cheat.setEnabled(enabled);
      GameCheats.addCheat(cheat, true);
      GameCheats.save();
      GameCheats.updateMachine();
    }
  }

  @Override
  @ApiMethod(156)
  public void removeProActionRocky(final String proActionRockyCode) {
    final Cheat cheat = ProActionRocky.convert(proActionRockyCode);
    if (cheat != null && GameCheats.removeCheat(cheat)) {
      GameCheats.save();
      GameCheats.updateMachine();
    }
  }

  @Override
  @ApiMethod(157)
  public int getPrgRomSize() {
    final Mapper mapper = _mapper;
    if (mapper != null) {
      return mapper.getPrgRomLength();
    } else {
      return 0;
    }
  }

  @Override
  @ApiMethod(158)
  public int readPrgRom(final int index) {
    final Mapper mapper = _mapper;
    if (mapper != null) {
      return mapper.readPrgRom(index);
    } else {
      return -1;
    }
  }

  @Override
  @ApiMethod(159)
  public void writePrgRom(final int index, final int value) {
    final Mapper mapper = _mapper;
    if (mapper != null) {
      mapper.writePrgRom(index, value & 0xFF);
    }
  }

  @Override
  @ApiMethod(160)
  public int getChrRomSize() {
    final Mapper mapper = _mapper;
    if (mapper != null) {
      return mapper.getChrRomLength();
    } else {
      return 0;
    }
  }

  @Override
  @ApiMethod(161)
  public int readChrRom(final int index) {
    final Mapper mapper = _mapper;
    if (mapper != null) {
      return mapper.readChrRom(index);
    } else {
      return -1;
    }
  }

  @Override
  @ApiMethod(162)
  public void writeChrRom(final int index, final int value) {
    final Mapper mapper = _mapper;
    if (mapper != null) {
      mapper.writeChrRom(index, value & 0xFF);
    }
  }

  @Override
  @ApiMethod(163)
  public int getStringWidth(final String str, final boolean monospaced) {
    return FontUtil.getWidth(str, monospaced);    
  }

  @Override
  @ApiMethod(164)
  public int getCharsWidth(final char[] chars, final boolean monospaced) {
    return FontUtil.getWidth(chars, monospaced);
  }
}
