using Nintaco

mutable struct TetrisBot
  api::RemoteAPI
  ai::AI
  playfieldUtil::PlayfieldUtil
  tetriminos::Array{Int,1}
  playfield::Array{Array{Int,1},1}
  TetriminosTypes::Array{Int,1}
  playFast::Bool
  playingDelay::Int
  targetTetriminoY::Int
  startCounter::Int
  movesIndex::Int
  moving::Bool
  states::Union{Array{State,1},Nothing}
  
  function TetrisBot(playFast)
    self = new()
    self.api = ApiSource.getAPI()
    self.ai = AI()
    self.playfieldUtil = PlayfieldUtil()
    self.tetriminos = Array{Int,1}(undef, TETRIMINOS_SEARCHED)
    self.playfield = createPlayfield(self.playfieldUtil)
    self.TetriminosTypes = Array{Int,1}(undef, 19)
    self.playFast = playFast
    self.playingDelay = 0
    self.targetTetriminoY = 0
    self.startCounter = 0
    self.movesIndex = 0
    self.moving = false
    self.states = nothing
    return self
  end
end

function launch(self)
  addActivateListener(self.api, () -> apiEnabled(self))
  addAccessPointListener(self.api, (type, address, value) 
      -> updateScore(self, type, address, value), AccessPointType.PreExecute, 
          0x9C35)
  addAccessPointListener(self.api, (type, address, value) 
      -> speedUpDrop(self, type, address, value), AccessPointType.PreExecute, 
          0x8977)
  addAccessPointListener(self.api, (type, address, tetriminoY) 
      -> tetriminoYUpdated(self, type, address, tetriminoY), 
          AccessPointType.PreWrite, Addresses.TetriminoY1)
  addAccessPointListener(self.api, (type, address, tetriminoY) 
      -> tetriminoYUpdated(self, type, address, tetriminoY), 
          AccessPointType.PreWrite, Addresses.TetriminoY2)
  addFrameListener(self.api, () -> renderFinished(self))
  addStatusListener(self.api, message -> statusChanged(self, message))
  Nintaco.run(self.api)
end

function apiEnabled(self)
  readTetriminoTypes(self)
end

function tetriminoYUpdated(self, type, address, tetriminoY)  
  if tetriminoY == 0
    self.targetTetriminoY = 0
  end
  if self.moving
    return self.targetTetriminoY
  else
    return tetriminoY
  end
end

function readTetriminoTypes(self)
  for i = 1:19
    self.TetriminosTypes[i] = 1 + readCPU(self.api, 
        Addresses.TetriminoTypeTable + i - 1)
  end
end  
  
function resetPlayState(self, gameState)
  if gameState != 4
    writeCPU(self.api, Addresses.PlayState, 0)
  end
end
  
function updateScore(self, type, address, value)
  # cap the points multiplier at 30 to avoid the kill screen
  if readCPU(self.api, 0x00A8) > 30
    writeCPU(self.api, 0x00A8, 30)
  end
  return -1
end

function speedUpDrop(self, type, address, value)
  setX(self.api, 0x1E)
  return -1
end

function setTetriminoYAddress(self, address, y)
  self.targetTetriminoY = y - 1
  writeCPU(self.api, address, y - 1)
end

function setTetriminoY(self, y)
  setTetriminoYAddress(self, Addresses.TetriminoY1, y)
  setTetriminoYAddress(self, Addresses.TetriminoY2, y)
end

function makeMove(self, tetriminoType, state, finalMove)
  if finalMove
    writeCPU(self.api, 0x006E, 0x03)
  end
  writeCPU(self.api, Addresses.TetriminoX, state.x - 1)
  setTetriminoY(self, state.y)
  writeCPU(self.api, Addresses.TetriminoID, 
      Tetriminos.ORIENTATIONS[tetriminoType][state.rotation].orientationID)
end  

function readTetrimino(self)
  return self.TetriminosTypes[1 + readCPU(self.api, Addresses.TetriminoID)]
end

function readNextTetrimino(self)
  return self.TetriminosTypes[1 + readCPU(self.api, Addresses.NextTetriminoID)]
end

function readPlayfield(self)
  self.tetriminos[1] = readTetrimino(self)
  self.tetriminos[2] = readNextTetrimino(self)
  
  for i = 1:PLAYFIELD_HEIGHT
    self.playfield[i][11] = 0
    for j = 1:PLAYFIELD_WIDTH
      if (readCPU(self.api, Addresses.Playfield + 10 * i + j - 11) 
          == EMPTY_SQUARE)
        self.playfield[i][j] = Tetriminos.NONE
      else
        self.playfield[i][j] = Tetriminos.I
        self.playfield[i][11] += 1
      end
    end
  end   
end

function spawned(self)
  currentTetrimino = readCPU(self.api, Addresses.TetriminoID)
  playState = readCPU(self.api, Addresses.PlayState)
  tetriminoX = readCPU(self.api, Addresses.TetriminoX)
  tetriminoY = readCPU(self.api, Addresses.TetriminoY1)
  
  return (playState == 1 && tetriminoX == 5 && tetriminoY == 0 
      && currentTetrimino < length(self.TetriminosTypes))
end

function isPlaying(self, gameState)
  return gameState == 4 && readCPU(self.api, Addresses.PlayState) < 9
end  

function pressStart(self)
  if self.startCounter > 0
    self.startCounter -= 1
  else
    self.startCounter = 10
  end 
  if self.startCounter >= 5
    writeGamepad(self.api, 0, GamepadButtons.Start, true)
  end    
end

function skipCopyrightScreen(self, gameState)
  if gameState == 0
    if readCPU(self.api, Addresses.Copyright1) > 1
      writeCPU(self.api,Addresses.Copyright1, 0)
    elseif readCPU(self.api, Addresses.Copyright2) > 2
      writeCPU(self.api, Addresses.Copyright2, 1)
    end
  end
end

function skipTitleAndDemoScreens(self, gameState)
  if gameState == 1 || gameState == 5
    pressStart(self)   
  else
    self.startCounter = 0
  end
end

function renderFinished(self)
  gameState = readCPU(self.api, Addresses.GameState)
  skipCopyrightScreen(self, gameState)
  skipTitleAndDemoScreens(self, gameState)
  resetPlayState(self, gameState)

  if isPlaying(self, gameState)
    if self.playingDelay > 0
      self.playingDelay -= 1
    elseif self.playFast
      # skip line clearing animation
      if readCPU(self.api, Addresses.PlayState) == 4
        writeCPU(self.api, Addresses.PlayState, 5)
      end
      if spawned(self)
        readPlayfield(self)
        state = search(self.ai, self.playfield, self.tetriminos)
        if state != nothing
          self.moving = true
          makeMove(self, self.tetriminos[1], state, true)
          self.moving = false
        end
      end
    else
      if self.moving && self.movesIndex <= length(self.states)
        makeMove(self, self.tetriminos[1], self.states[self.movesIndex],
            self.movesIndex == length(self.states))
        self.movesIndex += 1
      else
        self.moving = false
        if spawned(self)
          readPlayfield(self)
          state = search(self.ai, self.playfield, self.tetriminos)
          if state != nothing
            self.states = buildStatesList(self.ai, state)
            self.movesIndex = 1
            self.moving = true
          end
        end
      end
    end
  else
    self.states = nothing
    self.moving = false
    self.playingDelay = 16
  end 
end

function statusChanged(self, message)
  println(message)
end

function main(args...)
  ApiSource.initRemoteAPI("localhost", 9999)
  launch(TetrisBot(length(args) > 0 && "fast" == lowercase(args[1])))
end