using Printf
using Sockets

const EVENT_TYPES = [ 
  EventTypes.Activate, 
  EventTypes.Deactivate, 
  EventTypes.Stop, 
  EventTypes.Access, 
  EventTypes.Controllers, 
  EventTypes.Frame, 
  EventTypes.Scanline, 
  EventTypes.ScanlineCycle, 
  EventTypes.SpriteZero, 
  EventTypes.Status 
]

const EVENT_REQUEST = 0xFF
const EVENT_RESPONSE = 0xFE
const HEARTBEAT = 0xFD
const READY = 0xFC
const RETRY_SECONDS = 1.0

export RemoteAPI
mutable struct RemoteAPI
  listenerIDs::IdDict{Union{Function,AccessPoint,ScanlineCyclePoint,
      ScanlinePoint},Int}
  listenerObjects::Dict{Int,Dict{Int,Union{Function,AccessPoint,
      ScanlineCyclePoint,ScanlinePoint}}}
  host::String
  port::Int
  stream::Union{IO, Nothing}
  nextID::Int
  running::Bool
end

function RemoteAPI(host, port)
  listenerObjects = Dict{Int,Dict{Int,Union{Function,AccessPoint,
      ScanlineCyclePoint,ScanlinePoint}}}()
  for eventType in EVENT_TYPES
    listenerObjects[eventType] = Dict{Int,Union{Function,AccessPoint,
        ScanlineCyclePoint,ScanlinePoint}}()
  end
  return RemoteAPI(IdDict{Union{Function,AccessPoint,ScanlineCyclePoint,
      ScanlinePoint},Int}(), listenerObjects, host, port, nothing, 0, false)
end

export run
function run(api::RemoteAPI)
  if api.running
    return
  else
    api.running = true
  end
  while true
    fireStatusChanged(api, @sprintf "Connecting to %s:%d..." api.host api.port)
    try
      api.stream = connect(api.host, api.port)      
    catch
      fireStatusChanged(api, "Failed to establish connection.")
    end
    if api.stream != nothing
      try
        fireStatusChanged(api, "Connection established.")
        sendListeners(api)
        sendReady(api)
        while true
          probeEvents(api)
        end
      catch e
        if !isa(e, EOFError)
          println(e)
          println(stacktrace(catch_backtrace()))
        end
        fireDeactivated(api)
        fireStatusChanged(api, "Disconnected.")
      finally
        api.stream = nothing
      end
    end
    sleep(RETRY_SECONDS)
  end
end

function fireDeactivated(api)
  for listener in values(api.listenerObjects[EventTypes.Deactivate])
    listener()
  end
end

function fireStatusChanged(api, message)
  for listener in values(api.listenerObjects[EventTypes.Status])
    listener(message)
  end
end

function sendReady(api)
  if api.stream != nothing
    try
      writeByte(api.stream, READY)
      flush(api.stream)
    catch
    end
  end
end

function sendListeners(api)
  for (eventType, objs) in api.listenerObjects
    for (listenerID, listenerObject) in objs
      sendListener(api, listenerID, eventType, listenerObject)
    end
  end
end

function probeEvents(api)
      
  writeByte(api.stream, EVENT_REQUEST)
  flush(api.stream)
    
  eventType = readByte(api.stream)
    
  if eventType == HEARTBEAT
    writeByte(api.stream, EVENT_RESPONSE)
    flush(api.stream)
    return
  end
    
  listenerID = readInt(api.stream)
  obj = get(api.listenerObjects[eventType], listenerID, nothing)
    
  if obj != nothing
    if eventType == EventTypes.Access
      type = readInt(api.stream)
      address = readInt(api.stream)
      value = readInt(api.stream)
      result = obj.listener(type, address, value)
      writeByte(api.stream, EVENT_RESPONSE)
      writeInt(api.stream, result)
    else
      if (eventType == EventTypes.Activate
          || eventType == EventTypes.Deactivate
          || eventType == EventTypes.Stop
          || eventType == EventTypes.Controllers
          || eventType == EventTypes.Frame)
        obj()
      elseif eventType == EventTypes.Scanline
        obj.listener(readInt(api.stream))
      elseif eventType == EventTypes.ScanlineCycle
        scanline = readInt(api.stream)
        scanlineCycle = readInt(api.stream)
        address = readInt(api.stream)
        rendering = readBoolean(api.stream)
        obj.listener(scanline, scanlineCycle, address, rendering)
      elseif eventType == EventTypes.SpriteZero
        scanline = readInt(api.stream)
        scanlineCycle = readInt(api.stream)
        obj(scanline, scanlineCycle)
      elseif eventType == EventTypes.Status
        obj(readString(api.stream))
      else
        error("Unknown listener type: ", eventType)
      end 
      writeByte(api.stream, EVENT_RESPONSE)
    end
  end
    
  flush(api.stream)
end

function sendListener(api, listenerID, eventType, listenerObject)
  if api.stream != nothing
    try
      writeByte(api.stream, eventType)
      writeInt(api.stream, listenerID)
      if eventType == EventTypes.Access
        writeInt(api.stream, listenerObject.type)
        writeInt(api.stream, listenerObject.minAddress)
        writeInt(api.stream, listenerObject.maxAddress)
        writeInt(api.stream, listenerObject.bank)
      elseif eventType == EventTypes.Scanline
        writeInt(api.stream, listenerObject.scanline)
      elseif eventType == EventTypes.ScanlineCycle
        writeInt(api.stream, listenerObject.scanline)
        writeInt(api.stream, listenerObject.scanlineCycle)
      end      
      flush(api.stream)
    catch
    end
  end
end

function addListener(api, listener, eventType)
  if listener != nothing
    sendListener(api, addListenerObject(api, listener, eventType), eventType, 
        listener)
  end
end

export removeListener
function removeListener(api, listener, eventType, methodValue)
  if listener != nothing
    listenerID = removeListenerObject(api, listener, eventType)
    if listenerID >= 0 && api.stream != nothing
      try
        writeByte(api.stream, methodValue)
        writeInt(api.stream, listenerID)
        flush(api.stream)
      catch
      end
    end
  end
end

function addListenerObject(api, listener, eventType)
  return addListenerObject(api, listener, eventType, listener)
end

function addListenerObject(api, listener, eventType, listenerObject)
  listenerID = api.nextID += 1
  api.listenerIDs[listener] = listenerID
  api.listenerObjects[eventType][listenerID] = listenerObject
  return listenerID
end

function removeListenerObject(api, listener, eventType)
  listenerID = pop!(api.listenerIDs, listener, -1)
  if listenerID >= 0
    delete!(api.listenerObjects[eventType], listenerID)
  end
  return listenerID
end

export addActivateListener
function addActivateListener(api, listener)
  addListener(api, listener, EventTypes.Activate)
end

export removeActivateListener
function removeActivateListener(api, listener)
  removeListener(api, listener, EventTypes.Activate, 2)
end

export addDeactivateListener
function addDeactivateListener(api, listener)
  addListener(api, listener, EventTypes.Deactivate)
end

export removeDeactivateListener
function removeDeactivateListener(api, listener)
  removeListener(api, listener, EventTypes.Deactivate, 4)
end

export addStopListener
function addStopListener(api, listener)
  addListener(api, listener, EventTypes.Stop)
end

export removeStopListener
function removeStopListener(api, listener)
  removeListener(api, listener, EventTypes.Stop, 6)
end

export addAccessPointListener
function addAccessPointListener(api, listener, accessPointType, minAddress, 
    maxAddress = -1, bank = -1)  
  if listener != nothing
    point = AccessPoint(listener, accessPointType, minAddress, maxAddress, bank)
    sendListener(api, addListenerObject(api, listener, EventTypes.Access, 
        point), EventTypes.Access, point)
  end
end

export removeAccessPointListener
function removeAccessPointListener(api, listener)
  removeListener(api, listener, EventTypes.Access, 10)
end

export addControllersListener
function addControllersListener(api, listener)
  addListener(api, listener, EventTypes.Controllers)
end

export removeControllersListener
function removeControllersListener(api, listener)
  removeListener(api, listener, EventTypes.Controllers, 12)
end

export addFrameListener
function addFrameListener(api, listener)
  addListener(api, listener, EventTypes.Frame)
end

export removeFrameListener
function removeFrameListener(api, listener)
  removeListener(api, listener, EventTypes.Frame, 14)
end

export addScanlineListener
function addScanlineListener(api, listener, scanline)  
  if listener != nothing
    point = ScanlinePoint(listener, scanline)
    sendListener(api, addListenerObject(api, listener, EventTypes.Scanline, 
        point), EventTypes.Scanline, point)
  end
end

export removeScanlineListener
function removeScanlineListener(api, listener)
  removeListener(api, listener, EventTypes.Scanline, 16)
end

export addScanlineCycleListener
function addScanlineCycleListener(api, listener, scanline, scanlineCycle)  
  if listener != nothing
    point = ScanlineCyclePoint(listener, scanline, scanlineCycle)
    sendListener(api, addListenerObject(api, listener, EventTypes.ScanlineCycle, 
        point), EventTypes.ScanlineCycle, point)
  end
end

export removeScanlineCycleListener
function removeScanlineCycleListener(api, listener)
  removeListener(api, listener, EventTypes.ScanlineCycle, 18)
end

export addSpriteZeroListener
function addSpriteZeroListener(api, listener)
  addListener(api, listener, EventTypes.SpriteZero)
end

export removeSpriteZeroListener
function removeSpriteZeroListener(api, listener)
  removeListener(api, listener, EventTypes.SpriteZero, 20)
end

export addStatusListener
function addStatusListener(api, listener)
  addListener(api, listener, EventTypes.Status)
end

export removeStatusListener
function removeStatusListener(api, listener)
  removeListener(api, listener, EventTypes.Status, 22)
end

export getPixels
function getPixels(api, pixels)
  try
    writeByte(api.stream, 119)
    flush(api.stream)
    readIntArray(api.stream, pixels)
  catch
  end
end