﻿using System;
using System.Collections.Generic;
using System.Linq;
using Nintaco;

namespace Tetris {

  public delegate bool IChildFilter(int[][] playfield, int tetriminoType, int x, 
      int y, int rotation);
  public delegate void ISearchListener(int[][] playfield, int tetriminoType,
      int id, State state);

  public class State {
    public int x;
    public int y;
    public int rotation;
    public int visited;
    public State predecessor;
    public State next;

    public State(int x, int y, int rotation) {
      this.x = x;
      this.y = y;
      this.rotation = rotation;
    }
  }

  public class Orientation {
    public Point[] squares = new Point[4];
    public int minX;
    public int maxX;
    public int maxY;
    public int orientationID;

    public Orientation() {
      for (int i = 0; i < 4; i++) {
        squares[i] = new Point();
      }
    }
  }

  public class Point {

    public int x;
    public int y;

    public Point() {
    }

    public Point(int x, int y) {
      this.x = x;
      this.y = y;
    }
  }

  public class PlayfieldEvaluation {
    public int holes;
    public int columnTransitions;
    public int rowTransitions;
    public int wells;
  }

  public class Queue {

    private State head;
    private State tail;

    public void enqueue(State state) {
      if (head == null) {
        head = state;
        tail = state;
      } else {
        tail.next = state;
        tail = state;
      }
      state.next = null;
    }

    public State dequeue() {
      State state = head;
      if (head != null) {
        if (head == tail) {
          head = null;
          tail = null;
        } else {
          head = head.next;
        }
      }
      return state;
    }

    public bool isEmpty() {
      return head == null;
    }

    public bool isNotEmpty() {
      return head != null;
    }
  }

  public static class Tetriminos {

    public const int NONE = -1;
    public const int T = 0;
    public const int J = 1;
    public const int Z = 2;
    public const int O = 3;
    public const int S = 4;
    public const int L = 5;
    public const int I = 6;

    public static readonly int[][][,] PATTERNS = {
      new int[][,] {
      new int[,] { { -1,  0 }, {  0,  0 }, {  1,  0 }, {  0,  1 }, },    // Td*
      new int[,] { {  0, -1 }, { -1,  0 }, {  0,  0 }, {  0,  1 }, },    // Tl    
      new int[,] { { -1,  0 }, {  0,  0 }, {  1,  0 }, {  0, -1 }, },    // Tu
      new int[,] { {  0, -1 }, {  0,  0 }, {  1,  0 }, {  0,  1 }, }, }, // Tr   

      new int[][,] {
      new int[,] { { -1,  0 }, {  0,  0 }, {  1,  0 }, {  1,  1 }, },    // Jd*
      new int[,] { {  0, -1 }, {  0,  0 }, { -1,  1 }, {  0,  1 }, },    // Jl
      new int[,] { { -1, -1 }, { -1,  0 }, {  0,  0 }, {  1,  0 }, },    // Ju
      new int[,] { {  0, -1 }, {  1, -1 }, {  0,  0 }, {  0,  1 }, }, }, // Jr   

      new int[][,] {
      new int[,] { { -1,  0 }, {  0,  0 }, {  0,  1 }, {  1,  1 }, },    // Zh*
      new int[,] { {  1, -1 }, {  0,  0 }, {  1,  0 }, {  0,  1 }, }, }, // Zv   

      new int[][,] {
      new int[,] { { -1,  0 }, {  0,  0 }, { -1,  1 }, {  0,  1 }, }, }, // O*

      new int[][,] {
      new int[,] { {  0,  0 }, {  1,  0 }, { -1,  1 }, {  0,  1 }, },    // Sh*
      new int[,] { {  0, -1 }, {  0,  0 }, {  1,  0 }, {  1,  1 }, }, }, // Sv   

      new int[][,] {
      new int[,] { { -1,  0 }, {  0,  0 }, {  1,  0 }, { -1,  1 }, },    // Ld*
      new int[,] { { -1, -1 }, {  0, -1 }, {  0,  0 }, {  0,  1 }, },    // Ll
      new int[,] { {  1, -1 }, { -1,  0 }, {  0,  0 }, {  1,  0 }, },    // Lu
      new int[,] { {  0, -1 }, {  0,  0 }, {  0,  1 }, {  1,  1 }, }, }, // Lr      

      new int[][,] {
      new int[,] { { -2,  0 }, { -1,  0 }, {  0,  0 }, {  1,  0 }, },    // Ih*
      new int[,] { {  0, -2 }, {  0, -1 }, {  0,  0 }, {  0,  1 }, }, }, // Iv      
    };

    public static readonly int[] ORIENTATION_IDS
        = { 0x02, 0x03, 0x00, 0x01, 0x07, 0x04, 0x05, 0x06, 0x08, 0x09,
          0x0A, 0x0B, 0x0C, 0x0E, 0x0F, 0x10, 0x0D, 0x12, 0x11 };

    public static readonly Orientation[][] ORIENTATIONS 
        = new Orientation[PATTERNS.Length][];
  
    static Tetriminos() {
      for(int i = 0, idIndex = 0; i < PATTERNS.Length; i++) {
        List<Orientation> tetriminos = new List<Orientation>();
        for(int j = 0; j < PATTERNS[i].Length; j++) {
          Orientation tetrimino = new Orientation();
          tetriminos.Add(tetrimino);
          int minX = Int32.MaxValue;
          int maxX = Int32.MinValue;
          int maxY = Int32.MinValue;
          for(int k = 0; k < 4; k++) {
            int[,] p = PATTERNS[i][j];
            tetrimino.squares[k].x = p[k, 0];
            tetrimino.squares[k].y = p[k, 1];
            minX = Math.Min(minX, p[k, 0]);
            maxX = Math.Max(maxX, p[k, 0]);
            maxY = Math.Max(maxY, p[k, 1]);          
          }
          tetrimino.minX = -minX;
          tetrimino.maxX = AI.PLAYFIELD_WIDTH - maxX - 1;
          tetrimino.maxY = AI.PLAYFIELD_HEIGHT - maxY - 1;
          tetrimino.orientationID = ORIENTATION_IDS[idIndex++];
        }
        ORIENTATIONS[i] = tetriminos.ToArray<Orientation>();
      }
    }
  }

  public class Searcher {

    private static int globalMark = 1;

    private State[,,] states;
    private Queue queue = new Queue();
    private ISearchListener searchListener;
    private IChildFilter positionValidator;

    public Searcher(
        ISearchListener searchListener, IChildFilter positionValidator) {
      this.searchListener = searchListener;
      this.positionValidator = positionValidator;
      createStates();
    }

    private void createStates() {
      states = new State[AI.PLAYFIELD_HEIGHT, AI.PLAYFIELD_WIDTH, 4];
      for (int y = 0; y < AI.PLAYFIELD_HEIGHT; y++) {
        for (int x = 0; x < AI.PLAYFIELD_WIDTH; x++) {
          for (int rotation = 0; rotation < 4; rotation++) {
            states[y, x, rotation] = new State(x, y, rotation);
          }
        }
      }
    }

    private void lockTetrimino(
        int[][] playfield, int tetriminoType, int id, State state) {
      Point[] squares = Tetriminos.ORIENTATIONS[tetriminoType][state.rotation]
          .squares;
      for (int i = 0; i < 4; i++) {
        Point square = squares[i];
        int y = state.y + square.y;
        if (y >= 0) {
          playfield[y][state.x + square.x] = tetriminoType;
          playfield[y][AI.PLAYFIELD_WIDTH]++;
        }
      }
      searchListener(playfield, tetriminoType, id, state);
      for (int i = 0; i < 4; i++) {
        Point square = squares[i];
        int y = state.y + square.y;
        if (y >= 0) {
          playfield[y][state.x + square.x] = Tetriminos.NONE;
          playfield[y][AI.PLAYFIELD_WIDTH]--;
        }
      }
    }

    // returns true if the position is valid even if the node is not enqueued
    private bool addChild(int[][] playfield, int tetriminoType, int mark,
        State state, int x, int y, int rotation) {

      Orientation orientation = Tetriminos.ORIENTATIONS[tetriminoType]
          [rotation];
      if (x < orientation.minX || x > orientation.maxX 
          || y > orientation.maxY) {        
        return false;
      }

      State childNode = states[y, x, rotation];
      if (childNode.visited == mark) {
        return true;
      }

      Point[] squares = orientation.squares;
      for (int i = 0; i < 4; i++) {
        Point square = squares[i];
        int playfieldY = y + square.y;
        if (playfieldY >= 0
            && playfield[playfieldY][x + square.x] != Tetriminos.NONE) {
          return false;
        }
      }

      if (positionValidator != null && !positionValidator(playfield, 
          tetriminoType, x, y, rotation)) {
        return true;
      }

      childNode.visited = mark;
      childNode.predecessor = state;

      queue.enqueue(childNode);
      return true;
    }

    public bool search(int[][] playfield, int tetriminoType, int id) {

      int maxRotation = Tetriminos.ORIENTATIONS[tetriminoType].Length - 1;

      int mark = globalMark++;

      if (!addChild(playfield, tetriminoType, mark, null, 5, 0, 0)) {
        return false;
      }

      while (queue.isNotEmpty()) {

        State state = queue.dequeue();

        if (maxRotation != 0) {
          addChild(playfield, tetriminoType, mark, state, state.x, state.y,
              state.rotation == 0 ? maxRotation : state.rotation - 1);
          if (maxRotation != 1) {
            addChild(playfield, tetriminoType, mark, state, state.x, state.y,
                state.rotation == maxRotation ? 0 : state.rotation + 1);
          }
        }

        addChild(playfield, tetriminoType, mark, state,
            state.x - 1, state.y, state.rotation);
        addChild(playfield, tetriminoType, mark, state,
            state.x + 1, state.y, state.rotation);

        if (!addChild(playfield, tetriminoType, mark, state,
            state.x, state.y + 1, state.rotation)) {
          lockTetrimino(playfield, tetriminoType, id, state);
        }
      }

      return true;
    }
  }

  public class PlayfieldUtil {

    private readonly int[][] spareRows
        = new int[8 * AI.TETRIMINOS_SEARCHED][];
    private readonly int[] columnDepths = new int[AI.PLAYFIELD_WIDTH];
    private int spareIndex;

    public PlayfieldUtil() {
      for (int y = 0; y < spareRows.Length; y++) {
        spareRows[y] = new int[AI.PLAYFIELD_WIDTH + 1];
        for (int x = 0; x < AI.PLAYFIELD_WIDTH; x++) {
          spareRows[y][x] = Tetriminos.NONE;
        }
      }
    }

    public int[][] createPlayfield() {
      int[][] playfield = new int[AI.PLAYFIELD_HEIGHT][];
      for (int y = 0; y < AI.PLAYFIELD_HEIGHT; y++) {
        playfield[y] = new int[AI.PLAYFIELD_WIDTH + 1];
        for (int x = 0; x < AI.PLAYFIELD_WIDTH; x++) {
          playfield[y][x] = Tetriminos.NONE;
        }
      }
      return playfield;
    }

    public void lockTetrimino(
        int[][] playfield, int tetriminoType, State state) {

      Point[] squares = Tetriminos.ORIENTATIONS[tetriminoType][state.rotation]
          .squares;
      for (int i = 0; i < 4; i++) {
        Point square = squares[i];
        int y = state.y + square.y;
        if (y >= 0) {
          playfield[y][state.x + square.x] = tetriminoType;
          playfield[y][AI.PLAYFIELD_WIDTH]++;
        }
      }

      int startRow = state.y - 2;
      int endRow = state.y + 1;

      if (startRow < 1) {
        startRow = 1;
      }
      if (endRow >= AI.PLAYFIELD_HEIGHT) {
        endRow = AI.PLAYFIELD_HEIGHT - 1;
      }

      for (int y = startRow; y <= endRow; y++) {
        if (playfield[y][AI.PLAYFIELD_WIDTH] == AI.PLAYFIELD_WIDTH) {
          int[] clearedRow = playfield[y];
          for (int i = y; i > 0; i--) {
            playfield[i] = playfield[i - 1];
          }
          for (int x = 0; x < AI.PLAYFIELD_WIDTH; x++) {
            clearedRow[x] = Tetriminos.NONE;
          }
          clearedRow[AI.PLAYFIELD_WIDTH] = 0;
          playfield[0] = clearedRow;
        }
      }
    }

    public void evaluatePlayfield(
        int[][] playfield, PlayfieldEvaluation e) {

      for (int x = 0; x < AI.PLAYFIELD_WIDTH; x++) {
        columnDepths[x] = AI.PLAYFIELD_HEIGHT - 1;
        for (int y = 0; y < AI.PLAYFIELD_HEIGHT; y++) {
          if (playfield[y][x] != Tetriminos.NONE) {
            columnDepths[x] = y;
            break;
          }
        }
      }

      e.wells = 0;
      for (int x = 0; x < AI.PLAYFIELD_WIDTH; x++) {
        int minY = 0;
        if (x == 0) {
          minY = columnDepths[1];
        } else if (x == AI.PLAYFIELD_WIDTH - 1) {
          minY = columnDepths[AI.PLAYFIELD_WIDTH - 2];
        } else {
          minY = Math.Max(columnDepths[x - 1], columnDepths[x + 1]);
        }
        for (int y = columnDepths[x]; y >= minY; y--) {
          if ((x == 0 || playfield[y][x - 1] != Tetriminos.NONE)
              && (x == AI.PLAYFIELD_WIDTH - 1
                  || playfield[y][x + 1] != Tetriminos.NONE)) {
            e.wells++;
          }
        }
      }

      e.holes = 0;
      e.columnTransitions = 0;
      for (int x = 0; x < AI.PLAYFIELD_WIDTH; x++) {
        bool solid = true;
        for (int y = columnDepths[x] + 1; y < AI.PLAYFIELD_HEIGHT; y++) {
          if (playfield[y][x] == Tetriminos.NONE) {
            if (playfield[y - 1][x] != Tetriminos.NONE) {
              e.holes++;
            }
            if (solid) {
              solid = false;
              e.columnTransitions++;
            }
          } else if (!solid) {
            solid = true;
            e.columnTransitions++;
          }
        }
      }

      e.rowTransitions = 0;
      for (int y = 0; y < AI.PLAYFIELD_HEIGHT; y++) {
        bool solidFound = false;
        bool solid = true;
        int transitions = 0;
        for (int x = 0; x <= AI.PLAYFIELD_WIDTH; x++) {
          if (x == AI.PLAYFIELD_WIDTH) {
            if (!solid) {
              transitions++;
            }
          } else {
            if (playfield[y][x] == Tetriminos.NONE) {
              if (solid) {
                solid = false;
                transitions++;
              }
            } else {
              solidFound = true;
              if (!solid) {
                solid = true;
                transitions++;
              }
            }
          }
        }
        if (solidFound) {
          e.rowTransitions += transitions;
        }
      }
    }

    public int clearRows(int[][] playfield, int tetriminoY) {

      int rows = 0;
      int startRow = tetriminoY - 2;
      int endRow = tetriminoY + 1;

      if (startRow < 1) {
        startRow = 1;
      }
      if (endRow >= AI.PLAYFIELD_HEIGHT) {
        endRow = AI.PLAYFIELD_HEIGHT - 1;
      }

      for (int y = startRow; y <= endRow; y++) {
        if (playfield[y][AI.PLAYFIELD_WIDTH] == AI.PLAYFIELD_WIDTH) {
          rows++;
          clearRow(playfield, y);
        }
      }

      return rows;
    }

    private void clearRow(int[][] playfield, int y) {

      int[] clearedRow = playfield[y];
      clearedRow[AI.PLAYFIELD_WIDTH] = y;
      for (int i = y; i > 0; i--) {
        playfield[i] = playfield[i - 1];
      }
      playfield[0] = spareRows[spareIndex];
      playfield[0][AI.PLAYFIELD_WIDTH] = 0;

      spareRows[spareIndex++] = clearedRow;
    }

    private void restoreRow(int[][] playfield) {

      int[] restoredRow = spareRows[--spareIndex];
      int y = restoredRow[AI.PLAYFIELD_WIDTH];

      spareRows[spareIndex] = playfield[0];

      for (int i = 0; i < y; i++) {
        playfield[i] = playfield[i + 1];
      }
      restoredRow[AI.PLAYFIELD_WIDTH] = AI.PLAYFIELD_WIDTH;
      playfield[y] = restoredRow;
    }

    public void restoreRows(int[][] playfield, int rows) {
      for (int i = 0; i < rows; i++) {
        restoreRow(playfield);
      }
    }
  }

  public class AI {

    public const int PLAYFIELD_WIDTH = 10;
    public const int PLAYFIELD_HEIGHT = 20;
    public const int TETRIMINOS_SEARCHED = 2;

    private static readonly double[] WEIGHTS = {
      1.0,
      12.885008263218383,
      15.842707182438396,
      26.89449650779595,
      27.616914062397015,
      30.18511071927904,
    };

    private Searcher[] searchers;
    private int[] tetriminoIndices;
    private PlayfieldUtil playfieldUtil = new PlayfieldUtil();
    private PlayfieldEvaluation e = new PlayfieldEvaluation();
    private int totalRows;
    private int totalDropHeight;
    private double bestFitness;
    private State bestResult;
    private State result0;

    private ISearchListener searchListener;

    private void handleResult(int[][] playfield, int tetriminoType, int id, 
        State state) {

      if (id == 0) {
        result0 = state;
      }

      Orientation orientation
          = Tetriminos.ORIENTATIONS[tetriminoType][state.rotation];
      int rows = playfieldUtil.clearRows(playfield, state.y);
      int originalTotalRows = totalRows;
      int originalTotalDropHeight = totalDropHeight;
      totalRows += rows;
      totalDropHeight += orientation.maxY - state.y;

      int nextID = id + 1;

      if (nextID == tetriminoIndices.Length) {
        playfieldUtil.evaluatePlayfield(playfield, e);

        double fitness = computeFitness();
        if (fitness < bestFitness) {
          bestFitness = fitness;
          bestResult = result0;
        }
      } else {
        searchers[nextID].search(playfield, tetriminoIndices[nextID], nextID);
      }

      totalDropHeight = originalTotalDropHeight;
      totalRows = originalTotalRows;
      playfieldUtil.restoreRows(playfield, rows);
    }

    public AI() : this(null) { 
    }

    public AI(IChildFilter positionValidator) {
      searchListener = handleResult;
      searchers = new Searcher[AI.TETRIMINOS_SEARCHED];
      for (int i = 0; i < AI.TETRIMINOS_SEARCHED; i++) {
        searchers[i] = new Searcher(searchListener, positionValidator);
      }
    }

    private double computeFitness() {
      return WEIGHTS[0] * totalRows
           + WEIGHTS[1] * totalDropHeight
           + WEIGHTS[2] * e.wells
           + WEIGHTS[3] * e.holes
           + WEIGHTS[4] * e.columnTransitions
           + WEIGHTS[5] * e.rowTransitions;
    }

    public State search(int[][] playfield, int[] tetriminoIndices) {

      this.tetriminoIndices = tetriminoIndices;
      bestResult = null;
      bestFitness = Double.MaxValue;

      searchers[0].search(playfield, tetriminoIndices[0], 0);

      return bestResult;
    }

    public State[] buildStatesList(State state) {
      State s = state;
      int count = 0;
      while (s != null) {
        count++;
        s = s.predecessor;
      }
      State[] states = new State[count];
      while (state != null) {
        states[--count] = state;
        state = state.predecessor;
      }
      return states;
    }
  }



  public static class Addresses {
    public const int OrientationTable = 0x8A9C;
    public const int TetriminoTypeTable = 0x993B;
    public const int SpawnTable = 0x9956;
    public const int Copyright1 = 0x00C3;
    public const int Copyright2 = 0x00A8;
    public const int GameState = 0x00C0;
    public const int LowCounter = 0x00B1;
    public const int HighCounter = 0x00B2;
    public const int TetriminoX = 0x0060;
    public const int TetriminoY1 = 0x0061;
    public const int TetriminoY2 = 0x0041;
    public const int TetriminoID = 0x0062;
    public const int NextTetriminoID = 0x00BF;
    public const int FallTimer = 0x0065;
    public const int Playfield = 0x0400;
    public const int Level = 0x0064;
    public const int LevelTableAccess = 0x9808;
    public const int LinesHigh = 0x0071;
    public const int LinesLow = 0x0070;
    public const int PlayState = 0x0068;
  }

  public class TetrisBot {

    private static readonly int EMPTY_SQUARE = 0xEF;

    private readonly RemoteAPI api = ApiSource.API; 
  
    private readonly AI ai = new AI();
    private readonly PlayfieldUtil playfieldUtil = new PlayfieldUtil();
    private readonly int[] tetriminos = new int[AI.TETRIMINOS_SEARCHED];
    private readonly int[][] playfield;
    private readonly int[] TetriminosTypes = new int[19];
    private readonly bool playFast;  
  
    private int playingDelay;
    private int targetTetriminoY;
    private int startCounter;
    private int movesIndex;
    private bool moving;
    private State[] states;

    public TetrisBot(bool playFast) {
      this.playFast = playFast;
      playfield = playfieldUtil.createPlayfield();      
    }

    public void launch() {
      api.addActivateListener(apiEnabled);
      api.addAccessPointListener(updateScore, AccessPointType.PreExecute, 
          0x9C35);
      api.addAccessPointListener(speedUpDrop, AccessPointType.PreExecute, 
          0x8977);
      api.addAccessPointListener(tetriminoYUpdated, 
          AccessPointType.PreWrite, Addresses.TetriminoY1);
      api.addAccessPointListener(tetriminoYUpdated, 
          AccessPointType.PreWrite, Addresses.TetriminoY2);
      api.addFrameListener(renderFinished);
      api.addStatusListener(statusChanged);
      api.run();
    }

    private void apiEnabled() {
      readTetriminoTypes();
    }

    private int tetriminoYUpdated(int type, int address, int tetriminoY) {

      if (tetriminoY == 0) {
        targetTetriminoY = 0;
      }
      if (moving) {
        return targetTetriminoY;
      } else {
        return tetriminoY;
      }
    }

    private void readTetriminoTypes() {
      for (int i = 0; i < 19; i++) {
        TetriminosTypes[i] = api.readCPU(Addresses.TetriminoTypeTable + i);
      }
    }

    private void resetPlayState(int gameState) {
      if (gameState != 4) {
        api.writeCPU(Addresses.PlayState, 0);
      }
    }

    private int updateScore(int type, int address, int value) {
      // cap the points multiplier at 30 to avoid the kill screen
      if (api.readCPU(0x00A8) > 30) {
        api.writeCPU(0x00A8, 30);
      }
      return -1;
    }

    private int speedUpDrop(int type, int address, int value) {
      api.setX(0x1E);
      return -1;
    }

    private void setTetriminoYAddress(int address, int y) {
      targetTetriminoY = y;
      api.writeCPU(address, y);
    }

    private void setTetriminoY(int y) {
      setTetriminoYAddress(Addresses.TetriminoY1, y);
      setTetriminoYAddress(Addresses.TetriminoY2, y);
    }

    private void makeMove(int tetriminoType, State state, bool finalMove) {

      if (finalMove) {
        api.writeCPU(0x006E, 0x03);
      }
      api.writeCPU(Addresses.TetriminoX, state.x);
      setTetriminoY(state.y);
      api.writeCPU(Addresses.TetriminoID,
          Tetriminos.ORIENTATIONS[tetriminoType][state.rotation].orientationID);
    }

    private int readTetrimino() {
      return TetriminosTypes[api.readCPU(Addresses.TetriminoID)];
    }

    private int readNextTetrimino() {
      return TetriminosTypes[api.readCPU(Addresses.NextTetriminoID)];
    }

    private void readPlayfield() {
      tetriminos[0] = readTetrimino();
      tetriminos[1] = readNextTetrimino();

      for (int i = 0; i < AI.PLAYFIELD_HEIGHT; i++) {
        playfield[i][10] = 0;
        for (int j = 0; j < AI.PLAYFIELD_WIDTH; j++) {
          if (api.readCPU(Addresses.Playfield + 10 * i + j) == EMPTY_SQUARE) {
            playfield[i][j] = Tetriminos.NONE;
          } else {
            playfield[i][j] = Tetriminos.I;
            playfield[i][10]++;
          }
        }
      }
    }
      

    private bool spawned() {      
      int currentTetrimino = api.readCPU(Addresses.TetriminoID);
      int playState = api.readCPU(Addresses.PlayState);
      int tetriminoX = api.readCPU(Addresses.TetriminoX);
      int tetriminoY = api.readCPU(Addresses.TetriminoY1);

      return playState == 1 && tetriminoX == 5 && tetriminoY == 0
          && currentTetrimino < TetriminosTypes.Length;
    }

    private bool isPlaying(int gameState) {
      return gameState == 4 && api.readCPU(Addresses.PlayState) < 9;
    }

    private void pressStart() {
      if (startCounter > 0) {
        startCounter--;
      } else {
        startCounter = 10;
      }
      if (startCounter >= 5) {
        api.writeGamepad(0, GamepadButtons.Start, true);
      }
    }

    private void skipCopyrightScreen(int gameState) {
      if (gameState == 0) {
        if (api.readCPU(Addresses.Copyright1) > 1) {
          api.writeCPU(Addresses.Copyright1, 0);
        } else if (api.readCPU(Addresses.Copyright2) > 2) {
          api.writeCPU(Addresses.Copyright2, 1);
        }
      }
    }

    private void skipTitleAndDemoScreens(int gameState) {
      if (gameState == 1 || gameState == 5) {
        pressStart();
      } else {
        startCounter = 0;
      }
    }

    private void renderFinished() {
      int gameState = api.readCPU(Addresses.GameState);
      skipCopyrightScreen(gameState);
      skipTitleAndDemoScreens(gameState);
      resetPlayState(gameState);

      if (isPlaying(gameState)) {
        if (playingDelay > 0) {
          playingDelay--;
        } else if (playFast) {
          // skip line clearing animation
          if (api.readCPU(Addresses.PlayState) == 4) {
            api.writeCPU(Addresses.PlayState, 5);
          }
          if (spawned()) {
            readPlayfield();
            State state = ai.search(playfield, tetriminos);
            if (state != null) {
              moving = true;
              makeMove(tetriminos[0], state, true);
              moving = false;
            }
          }
        } else {
          if (moving && movesIndex < states.Length) {
            makeMove(tetriminos[0], states[movesIndex],
                movesIndex == states.Length - 1);
            movesIndex++;
          } else {
            moving = false;
            if (spawned()) {
              readPlayfield();
              State state = ai.search(playfield, tetriminos);
              if (state != null) {                
                states = ai.buildStatesList(state);
                movesIndex = 0;
                moving = true;
              }
            }
          }
        }
      } else {
        states = null;
        moving = false;
        playingDelay = 16;
      }
    }

    private void statusChanged(string message) {
      Console.WriteLine(message);      
    }

    public static void Main(string[] args) {
      ApiSource.initRemoteAPI("localhost", 9999);
      new TetrisBot(args.Length > 0 && "fast" == args[0].ToLower()).launch();
    }
  }
}
