#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>

#include "nintaco.h"

#define MIN(a,b) (((a)<(b))?(a):(b))
#define MAX(a,b) (((a)>(b))?(a):(b))

#define PLAYFIELD_WIDTH 10
#define PLAYFIELD_HEIGHT 20
#define TETRIMINOS_SEARCHED 2

const double WEIGHTS[] = {
  1.0,
  12.885008263218383,
  15.842707182438396,
  26.89449650779595,
  27.616914062397015,
  30.18511071927904,
};

typedef enum {
  NONE = -1,
  T = 0,
  J = 1,
  Z = 2,
  O = 3,
  S = 4,
  L = 5,
  I = 6,
} Tetriminos;

const int ORIENTATION_IDS[] 
      = { 0x02, 0x03, 0x00, 0x01, 0x07, 0x04, 0x05, 0x06, 0x08, 0x09, 
          0x0A, 0x0B, 0x0C, 0x0E, 0x0F, 0x10, 0x0D, 0x12, 0x11 };

const int PATTERN_LENGTHS[] = { 4, 4, 2, 1, 2, 4, 2 };

const int PATTERNS[7][4][4][2] = {
  { { { -1,  0 }, {  0,  0 }, {  1,  0 }, {  0,  1 }, },    // Td (spawn)
    { {  0, -1 }, { -1,  0 }, {  0,  0 }, {  0,  1 }, },    // Tl    
    { { -1,  0 }, {  0,  0 }, {  1,  0 }, {  0, -1 }, },    // Tu
    { {  0, -1 }, {  0,  0 }, {  1,  0 }, {  0,  1 }, }, }, // Tr   

  { { { -1,  0 }, {  0,  0 }, {  1,  0 }, {  1,  1 }, },    // Jd (spawn)
    { {  0, -1 }, {  0,  0 }, { -1,  1 }, {  0,  1 }, },    // Jl
    { { -1, -1 }, { -1,  0 }, {  0,  0 }, {  1,  0 }, },    // Ju
    { {  0, -1 }, {  1, -1 }, {  0,  0 }, {  0,  1 }, }, }, // Jr   

  { { { -1,  0 }, {  0,  0 }, {  0,  1 }, {  1,  1 }, },    // Zh (spawn) 
    { {  1, -1 }, {  0,  0 }, {  1,  0 }, {  0,  1 }, }, }, // Zv   

  { { { -1,  0 }, {  0,  0 }, { -1,  1 }, {  0,  1 }, }, }, // O  (spawn)   

  { { {  0,  0 }, {  1,  0 }, { -1,  1 }, {  0,  1 }, },    // Sh (spawn)
    { {  0, -1 }, {  0,  0 }, {  1,  0 }, {  1,  1 }, }, }, // Sv   

  { { { -1,  0 }, {  0,  0 }, {  1,  0 }, { -1,  1 }, },    // Ld (spawn)
    { { -1, -1 }, {  0, -1 }, {  0,  0 }, {  0,  1 }, },    // Ll
    { {  1, -1 }, { -1,  0 }, {  0,  0 }, {  1,  0 }, },    // Lu
    { {  0, -1 }, {  0,  0 }, {  0,  1 }, {  1,  1 }, }, }, // Lr      

  { { { -2,  0 }, { -1,  0 }, {  0,  0 }, {  1,  0 }, },    // Ih (spawn)    
    { {  0, -2 }, {  0, -1 }, {  0,  0 }, {  0,  1 }, }, }, // Iv      
};

typedef struct {
  int holes;
  int columnTransitions;
  int rowTransitions;
  int wells;
} PlayfieldEvaluation;

typedef struct {
  int x;
  int y;
} Point;

typedef struct {
  Point squares[4];
  int minX;
  int maxX;
  int maxY;
  int orientationID;    
} Orientation;

Orientation ORIENTATIONS[7][4];

void initOrientations() {
  int i;
  int idIndex;
  for(i = 0, idIndex = 0; i < 7; i++) {    
    for(int j = 0; j < PATTERN_LENGTHS[i]; j++) {
      int minX = 1000;
      int maxX = -1000;
      int maxY = -1000;
      for(int k = 0; k < 4; k++) {
        ORIENTATIONS[i][j].squares[k].x = PATTERNS[i][j][k][0];
        ORIENTATIONS[i][j].squares[k].y = PATTERNS[i][j][k][1];
        minX = MIN(minX, PATTERNS[i][j][k][0]);
        maxX = MAX(maxX, PATTERNS[i][j][k][0]);
        maxY = MAX(maxY, PATTERNS[i][j][k][1]);          
      }
      ORIENTATIONS[i][j].minX = -minX;
      ORIENTATIONS[i][j].maxX = PLAYFIELD_WIDTH - maxX - 1;
      ORIENTATIONS[i][j].maxY = PLAYFIELD_HEIGHT - maxY - 1;
      ORIENTATIONS[i][j].orientationID = ORIENTATION_IDS[idIndex++];
    }    
  }
}

typedef struct {
  int** spareRows; 
  int columnDepths[PLAYFIELD_WIDTH];
  int spareIndex;
} PlayfieldUtil;

typedef struct _State {
  int x;
  int y;
  int rotation;
  int32_t visited;
  struct _State* predecessor;
  struct _State* next;
} State;

typedef struct {
  State* head;
  State* tail;
} Queue;

int queueSize = 0; // TODO REMOVE

void enqueue(Queue* queue, State* state) {
  if (queue->head == NULL) {
    queue->head = state;
    queue->tail = state;        
  } else {
    queue->tail->next = state;
    queue->tail = state;        
  }
  state->next = NULL;
}

State* dequeue(Queue* queue) {
  State* state = queue->head;
  if (queue->head != NULL) {
    if (queue->head == queue->tail) {
      queue->head = NULL;
      queue->tail = NULL;
    } else {
      queue->head = queue->head->next;
    }
  }
  return state;
}

bool isEmpty(Queue* queue) { 
  return queue->head == NULL;
}

bool isNotEmpty(Queue* queue) { 
  return queue->head != NULL;
}  

struct _Searcher;
typedef struct _Searcher Searcher;

struct _AI;
typedef struct _AI AI;

typedef bool (*IChildFilter)(Searcher* searcher, int** playfield, 
    int tetriminoType, int x, int y, int rotation);
typedef void (*ISearchListener)(AI* ai, int** playfield, int tetriminoType, 
    int id, State* state);

struct _Searcher {
  State states[PLAYFIELD_HEIGHT][PLAYFIELD_WIDTH][4];
  Queue queue;
  ISearchListener searchListener;
  IChildFilter positionValidator;
};

struct _AI {
  Searcher searchers[2]; 
  int* tetriminoIndices;
  PlayfieldUtil* playfieldUtil;
  PlayfieldEvaluation e;
  int totalRows;
  int totalDropHeight;  
  double bestFitness;
  State* bestResult;
  State* result0;
};

void initState(State* state, int x, int y, int rotation) {
  state->x = x;
  state->y = y;
  state->rotation = rotation;
  state->next = NULL;
  state->predecessor = NULL;
  state->visited = 0;
}

void initQueue(Queue* queue) {
  queue->head = NULL;
  queue->tail = NULL;
}

void initSearcher(Searcher *searcher, ISearchListener searchListener, 
    IChildFilter positionValidator) {
  int x;
  int y;  
  searcher->searchListener = searchListener;
  searcher->positionValidator = positionValidator;  
  initQueue(&searcher->queue);
  for(int y = 0; y < PLAYFIELD_HEIGHT; y++) {
    for(int x = 0; x < PLAYFIELD_WIDTH; x++) {        
      for(int rotation = 0; rotation < 4; rotation++) { 
        initState(&searcher->states[y][x][rotation], x, y, rotation);
      }
    }
  }
}

void lockTetrimino(AI* ai, Searcher *searcher, int** playfield, 
    int tetriminoType, int id, State* state) {

  int i;
  Point* squares = ORIENTATIONS[tetriminoType][state->rotation].squares;
  for(i = 0; i < 4; i++) {
    Point* square = squares + i;
    int y = state->y + square->y;
    if (y >= 0) {
      playfield[y][state->x + square->x] = tetriminoType;
      playfield[y][PLAYFIELD_WIDTH]++;
    }
  }
    
  searcher->searchListener(ai, playfield, tetriminoType, id, state);
  for(i = 0; i < 4; i++) {
    Point* square = squares + i;
    int y = state->y + square->y;
    if (y >= 0) {
      playfield[y][state->x + square->x] = NONE;
      playfield[y][PLAYFIELD_WIDTH]--;
    }
  }
}

// returns true if the position is valid even if the node is not enqueued
bool addChild(Searcher *searcher, int** playfield, int tetriminoType, 
    int32_t mark, State* state, int x, int y, int rotation) {

  int i;
  
  Orientation* orientation = &ORIENTATIONS[tetriminoType][rotation];
  
  if (x < orientation->minX || x > orientation->maxX 
      || y > orientation->maxY) {
    return false;
  }

  State* childNode = &searcher->states[y][x][rotation];
  if (childNode->visited == mark) {
    return true;
  }

  Point* squares = orientation->squares;
  for(i = 0; i < 4; i++) {
    Point* square = squares + i;
    int playfieldY = y + square->y;
    if (playfieldY >= 0 && playfield[playfieldY][x + square->x] != NONE) {
      return false;
    }
  }

  if (searcher->positionValidator && !searcher->positionValidator(searcher,
      playfield, tetriminoType, x, y, rotation)) {
    return true;
  }

  childNode->visited = mark;
  childNode->predecessor = state;

  enqueue(&searcher->queue, childNode);   
  return true; 
}

bool search(AI* ai, Searcher *searcher, int** playfield, int tetriminoType, 
    int id) {
  
  static int32_t globalMark = 1;
  
  int maxRotation = PATTERN_LENGTHS[tetriminoType] - 1;
  int32_t mark = globalMark++;
  Queue* queue = &searcher->queue;

  if (!addChild(searcher, playfield, tetriminoType, mark, NULL, 5, 0, 0)) {
    return false;
  }    
  
  while(isNotEmpty(queue)) {
    State* state = dequeue(queue);

    if (maxRotation != 0) {
      addChild(searcher, playfield, tetriminoType, mark, state, state->x, 
          state->y, state->rotation == 0 ? maxRotation : state->rotation - 1);
      if (maxRotation != 1) {
        addChild(searcher, playfield, tetriminoType, mark, state, state->x, 
            state->y, state->rotation == maxRotation ? 0 : state->rotation + 1);
      }
    }
    
    addChild(searcher, playfield, tetriminoType, mark, state, state->x - 1, 
        state->y, state->rotation);
    addChild(searcher, playfield, tetriminoType, mark, state, state->x + 1, 
        state->y, state->rotation);

    if (!addChild(searcher, playfield, tetriminoType, mark, state,
        state->x, state->y + 1, state->rotation)) {
      lockTetrimino(ai, searcher, playfield, tetriminoType, id, state);
    }
  }

  return true;
}

PlayfieldUtil* newPlayfieldUtil() {
  int x;
  int y;  
  PlayfieldUtil* util = (PlayfieldUtil*)malloc(sizeof(PlayfieldUtil));
  util->spareRows = (int**)malloc(sizeof(int*) * 8 * TETRIMINOS_SEARCHED);
  for(y = 0; y < 8 * TETRIMINOS_SEARCHED; y++) {
    util->spareRows[y] = (int*)malloc(sizeof(int) * (PLAYFIELD_WIDTH + 1));
    for(x = 0; x < PLAYFIELD_WIDTH; x++) {
      util->spareRows[y][x] = NONE;
    }
    util->spareRows[y][PLAYFIELD_WIDTH] = 0;
  }  
  return util;
}

void deletePlayfieldUtil(PlayfieldUtil* util) {
  int y;
  for(y = 0; y < 8 * TETRIMINOS_SEARCHED; y++) {
    free(util->spareRows[y]);
  }
  free(util->spareRows);
  free(util);
}

int** newPlayfield() {
  int x;
  int y;
  int** playfield = (int**)malloc(sizeof(int*) * PLAYFIELD_HEIGHT);
  for(y = 0; y < PLAYFIELD_HEIGHT; y++) {
    playfield[y] = (int*)malloc(sizeof(int) * (PLAYFIELD_WIDTH + 1));
    for(x = 0; x < PLAYFIELD_WIDTH; x++) {
      playfield[y][x] = NONE;
    }
    playfield[y][PLAYFIELD_WIDTH] = 0;
  }
  return playfield;
}

void deletePlayfield(int** playfield) {
  int y;
  for(y = 0; y < PLAYFIELD_HEIGHT; y++) {
    free(playfield[y]);
  }
  free(playfield);
}

void lockTetriminoIntoPlayfield(int** playfield, int tetriminoType, 
    State* state) {
    
  int i;
  int x;
  int y;
  
  Point* squares = ORIENTATIONS[tetriminoType][state->rotation].squares;
  for(i = 0; i < 4; i++) {
    Point* square = squares + i;
    int y = state->y + square->y;
    if (y >= 0) {
      playfield[y][state->x + square->x] = tetriminoType;
      playfield[y][PLAYFIELD_WIDTH]++;
    }
  }

  int startRow = state->y - 2;
  int endRow = state->y + 1;

  if (startRow < 1) {
    startRow = 1;
  }
  if (endRow >= PLAYFIELD_HEIGHT) {
    endRow = PLAYFIELD_HEIGHT - 1;
  }

  for(y = startRow; y <= endRow; y++) {
    if (playfield[y][PLAYFIELD_WIDTH] == PLAYFIELD_WIDTH) { 
      int* clearedRow = playfield[y];
      for(i = y; i > 0; i--) {
        playfield[i] = playfield[i - 1];
      }        
      for(x = 0; x < PLAYFIELD_WIDTH; x++) {
        clearedRow[x] = NONE;          
      }
      clearedRow[PLAYFIELD_WIDTH] = 0;
      playfield[0] = clearedRow;
    }
  }
}

void evaluatePlayfield(PlayfieldUtil* util, int** playfield, 
    PlayfieldEvaluation* e) {
    
  int x;
  int y;
  int* columnDepths = util->columnDepths;
  
  for(x = 0; x < PLAYFIELD_WIDTH; x++) {
    columnDepths[x] = PLAYFIELD_HEIGHT - 1; // TODO THIS LINE IS FAILING :(
    for(y = 0; y < PLAYFIELD_HEIGHT; y++) {
      if (playfield[y][x] != NONE) {
        columnDepths[x] = y;
        break;
      }
    }      
  }

  e->wells = 0;
  for(x = 0; x < PLAYFIELD_WIDTH; x++) {
    int minY = 0;
    if (x == 0) {
      minY = columnDepths[1];
    } else if (x == PLAYFIELD_WIDTH - 1) {
      minY = columnDepths[PLAYFIELD_WIDTH - 2];
    } else {
      minY = MAX(columnDepths[x - 1], columnDepths[x + 1]);
    }
    for(y = columnDepths[x]; y >= minY; y--) {
      if ((x == 0 || playfield[y][x - 1] != NONE)
          && (x == PLAYFIELD_WIDTH - 1 || playfield[y][x + 1] != NONE)) {
        e->wells++;
      } 
    }
  }

  e->holes = 0;
  e->columnTransitions = 0;
  for(x = 0; x < PLAYFIELD_WIDTH; x++) {
    bool solid = true;
    for(y = columnDepths[x] + 1; y < PLAYFIELD_HEIGHT; y++) {      
      if (playfield[y][x] == NONE) {
        if (playfield[y - 1][x] != NONE) {
          e->holes++;
        }
        if (solid) {
          solid = false;
          e->columnTransitions++;
        }
      } else if (!solid) {
        solid = true;
        e->columnTransitions++;
      }       
    }
  }

  e->rowTransitions = 0;
  for(y = 0; y < PLAYFIELD_HEIGHT; y++) {
    bool solidFound = false;
    bool solid = true;
    int transitions = 0;
    for(x = 0; x <= PLAYFIELD_WIDTH; x++) {
      if (x == PLAYFIELD_WIDTH) {
        if (!solid) {
          transitions++;
        }
      } else {          
        if (playfield[y][x] == NONE) {            
          if (solid) {
            solid = false;
            transitions++;
          }
        } else {          
          solidFound = true;
          if (!solid) {
            solid = true;
            transitions++;                           
          }
        }
      }
    }
    if (solidFound) {        
      e->rowTransitions += transitions;
    }
  }    
}

void clearRow(PlayfieldUtil* util, int** playfield, int y) {
  int i;
  int* clearedRow = playfield[y];
  clearedRow[PLAYFIELD_WIDTH] = y;
  for(i = y; i > 0; i--) {
    playfield[i] = playfield[i - 1];
  }
  playfield[0] = util->spareRows[util->spareIndex];
  playfield[0][PLAYFIELD_WIDTH] = 0;

  util->spareRows[util->spareIndex++] = clearedRow;    
}

int clearRows(PlayfieldUtil* util, int** playfield, int tetriminoY) {

  int y;
  int rows = 0;
  int startRow = tetriminoY - 2;
  int endRow = tetriminoY + 1;

  if (startRow < 1) {
    startRow = 1;
  }    
  if (endRow >= PLAYFIELD_HEIGHT) {
    endRow = PLAYFIELD_HEIGHT - 1;
  }

  for(y = startRow; y <= endRow; y++) {
    if (playfield[y][PLAYFIELD_WIDTH] == PLAYFIELD_WIDTH) {
      rows++;
      clearRow(util, playfield, y);
    }
  }

  return rows;
}

void restoreRow(PlayfieldUtil* util, int** playfield) {

  int i;
  int* restoredRow = util->spareRows[--(util->spareIndex)];
  int y = restoredRow[PLAYFIELD_WIDTH];

  util->spareRows[util->spareIndex] = playfield[0];

  for(i = 0; i < y; i++) {
    playfield[i] = playfield[i + 1];
  }
  restoredRow[PLAYFIELD_WIDTH] = PLAYFIELD_WIDTH;
  playfield[y] = restoredRow;
}

void restoreRows(PlayfieldUtil* util, int** playfield, int rows) {    
  int i;
  for(i = 0; i < rows; i++) {
    restoreRow(util, playfield);
  }    
}

double computeFitness(AI* ai) {
  return WEIGHTS[0] * ai->totalRows
       + WEIGHTS[1] * ai->totalDropHeight
       + WEIGHTS[2] * ai->e.wells
       + WEIGHTS[3] * ai->e.holes                 
       + WEIGHTS[4] * ai->e.columnTransitions
       + WEIGHTS[5] * ai->e.rowTransitions;         
}

void searchListener(AI* ai, int** playfield, int tetriminoType, int id, 
    State* state) {
      
  if (id == 0) {
    ai->result0 = state;
  }
  
  int rows = clearRows(ai->playfieldUtil, playfield, state->y);
  int originalTotalRows = ai->totalRows;
  int originalTotalDropHeight = ai->totalDropHeight;
  ai->totalRows += rows;
  ai->totalDropHeight += ORIENTATIONS[tetriminoType][state->rotation].maxY 
      - state->y;

  int nextID = id + 1;
  
  if (nextID == TETRIMINOS_SEARCHED) {
    evaluatePlayfield(ai->playfieldUtil, playfield, &ai->e);
    double fitness = computeFitness(ai);
    if (fitness < ai->bestFitness) {
      ai->bestFitness = fitness;
      ai->bestResult = ai->result0;
    }
  } else {
    search(ai, &ai->searchers[nextID], playfield, ai->tetriminoIndices[nextID], 
        nextID);
  }

  ai->totalDropHeight = originalTotalDropHeight;
  ai->totalRows = originalTotalRows;
  restoreRows(ai->playfieldUtil, playfield, rows);
}

AI* newAI(IChildFilter positionValidator) {
  int i;
  AI* ai = (AI*)malloc(sizeof(AI));
  for(i = 0; i < TETRIMINOS_SEARCHED; i++) {
    initSearcher(&ai->searchers[i], &searchListener, positionValidator);    
  }
  ai->bestFitness = 0;
  ai->bestResult = NULL;
  ai->playfieldUtil = newPlayfieldUtil();
  ai->result0 = NULL;
  ai->tetriminoIndices = NULL;
  ai->totalDropHeight = 0;
  ai->totalRows = 0;
  return ai;
}

void deleteAI(AI* ai) {
  free(ai);
}

State* aiSearch(AI* ai, int** playfield, int* tetriminoIndices) {

  ai->tetriminoIndices = tetriminoIndices;
  ai->bestResult = NULL;
  ai->bestFitness = 1E20;

  search(ai, &ai->searchers[0], playfield, tetriminoIndices[0], 0);

  return ai->bestResult;
}  
  
void buildStatesList(State* state, State** stateList, int* stateListLength) {
  State* s = state;
  int count = 0;      
  while(s) {
    count++;
    s = s->predecessor;
  }
  *stateListLength = count;
  while(state) {
    stateList[--count] = state;
    state = state->predecessor;
  }
}

/* Nintaco API code follows. */

#define playFast false

#define EMPTY_SQUARE 0xEF

typedef enum {
  OrientationTable = 0x8A9C,
  TetriminoTypeTable = 0x993B,
  SpawnTable = 0x9956,
  Copyright1 = 0x00C3,
  Copyright2 = 0x00A8,
  GameState = 0x00C0,
  LowCounter = 0x00B1,
  HighCounter = 0x00B2,
  TetriminoX = 0x0060,
  TetriminoY1 = 0x0061,
  TetriminoY2 = 0x0041,
  TetriminoID = 0x0062,
  NextTetriminoID = 0x00BF,
  FallTimer = 0x0065,
  Playfield = 0x0400,
  Level = 0x0064,
  LevelTableAccess = 0x9808,
  LinesHigh = 0x0071,
  LinesLow = 0x0070,
  PlayState = 0x0068,  
} Addresses;

typedef struct {
  NintacoAPI* api;  
  AI *ai;
  PlayfieldUtil *playfieldUtil;
  int tetriminos[TETRIMINOS_SEARCHED];
  int** playfield;  
  int TetriminosTypes[19];    
  int playingDelay;
  int targetTetriminoY;
  int startCounter;
  int movesIndex;
  bool moving;
  State* states[1024];
  int statesLength;
} TetrisBot;

TetrisBot tetrisBot;

void apiEnabled(NintacoAPI* api) {
  for(int i = 0; i < 19; i++) {
    tetrisBot.TetriminosTypes[i] = readCPU(api, TetriminoTypeTable + i);
  }
}

int tetriminoYUpdated(NintacoAPI* api, int type, int address, int tetriminoY) {
  if (tetriminoY == 0) {
    tetrisBot.targetTetriminoY = 0;
  }
  if (tetrisBot.moving) {      
    return tetrisBot.targetTetriminoY;
  } else {
    return tetriminoY;
  }
}

void resetPlayState(int gameState) {
  if (gameState != 4) {
    writeCPU(tetrisBot.api, PlayState, 0);
  }
}

int updateScore(NintacoAPI* api, int type, int address, int value) {
  // cap the points multiplier at 30 to avoid the kill screen
  if (readCPU(tetrisBot.api, 0x00A8) > 30) {
    writeCPU(tetrisBot.api, 0x00A8, 30);
  }
  return -1;
}

int speedUpDrop(NintacoAPI* api, int type, int address, int value) {
  setX(tetrisBot.api, 0x1E);
  return -1;
}

void setTetriminoYAddress(int address, int y) {
  tetrisBot.targetTetriminoY = y;
  writeCPU(tetrisBot.api, address, y);
}

void setTetriminoY(int y) {
  setTetriminoYAddress(TetriminoY1, y);
  setTetriminoYAddress(TetriminoY2, y);
}

void makeMove(int tetriminoType, State* state, bool finalMove) {

  if (finalMove) { 
    writeCPU(tetrisBot.api, 0x006E, 0x03);
  }
  writeCPU(tetrisBot.api, TetriminoX, state->x);
  setTetriminoY(state->y);
  writeCPU(tetrisBot.api, TetriminoID, 
      ORIENTATIONS[tetriminoType][state->rotation].orientationID);
}  

int readTetrimino() {
  return tetrisBot.TetriminosTypes[readCPU(tetrisBot.api, TetriminoID)];
}

int readNextTetrimino() {
  return tetrisBot.TetriminosTypes[readCPU(tetrisBot.api, NextTetriminoID)];
}

void readPlayfield() {
  int i;
  int j;
  
  tetrisBot.tetriminos[0] = readTetrimino();
  tetrisBot.tetriminos[1] = readNextTetrimino();

  for(i = 0; i < PLAYFIELD_HEIGHT; i++) {
    tetrisBot.playfield[i][10] = 0;
    for (j = 0; j < PLAYFIELD_WIDTH; j++) {
      if (readCPU(tetrisBot.api, Playfield + 10 * i + j) == EMPTY_SQUARE) {
        tetrisBot.playfield[i][j] = NONE;
      } else {
        tetrisBot.playfield[i][j] = I;
        tetrisBot.playfield[i][10]++;
      }
    }
  }    
}

bool spawned() {
  int currentTetrimino = readCPU(tetrisBot.api, TetriminoID);
  int playState = readCPU(tetrisBot.api, PlayState);
  int tetriminoX = readCPU(tetrisBot.api, TetriminoX);
  int tetriminoY = readCPU(tetrisBot.api, TetriminoY1); 

  return playState == 1 && tetriminoX == 5 && tetriminoY == 0 
      && currentTetrimino < 19;
}

bool isPlaying(int gameState) {
  return gameState == 4 && readCPU(tetrisBot.api, PlayState) < 9;
}  

void pressStart() {
  if (tetrisBot.startCounter > 0) {
    tetrisBot.startCounter--;
  } else {
    tetrisBot.startCounter = 10;
  } 
  if (tetrisBot.startCounter >= 5) {
    writeGamepad(tetrisBot.api, 0, Start, true);
  }    
}

void skipCopyrightScreen(int gameState) {
  if (gameState == 0) {
    if (readCPU(tetrisBot.api, Copyright1) > 1) {
      writeCPU(tetrisBot.api, Copyright1, 0);
    } else if (readCPU(tetrisBot.api, Copyright2) > 2) {
      writeCPU(tetrisBot.api, Copyright2, 1);
    }
  }
}

void skipTitleAndDemoScreens(int gameState) {
  if (gameState == 1 || gameState == 5) {
    pressStart();   
  } else {
    tetrisBot.startCounter = 0;
  }
}

void renderFinished(NintacoAPI* api) {
  int gameState = readCPU(api, GameState);
  skipCopyrightScreen(gameState);
  skipTitleAndDemoScreens(gameState);
  resetPlayState(gameState);

  if (isPlaying(gameState)) {
    if (tetrisBot.playingDelay > 0) {
      tetrisBot.playingDelay--;
    } else if (playFast) {
      // skip line clearing animation
      if (readCPU(api, PlayState) == 4) {
        writeCPU(api, PlayState, 5);
      }
      if (spawned()) {
        readPlayfield();
        State* state = aiSearch(tetrisBot.ai, tetrisBot.playfield, 
            tetrisBot.tetriminos);
        if (state != NULL) {
          tetrisBot.moving = true;
          makeMove(tetrisBot.tetriminos[0], state, true);
          tetrisBot.moving = false;
        }
      }
    } else {
      if (tetrisBot.moving && tetrisBot.movesIndex < tetrisBot.statesLength) {
        makeMove(tetrisBot.tetriminos[0], 
            tetrisBot.states[tetrisBot.movesIndex],
                tetrisBot.movesIndex == tetrisBot.statesLength - 1);
        tetrisBot.movesIndex++;
      } else {   
        tetrisBot.moving = false;
        if (spawned()) {
          readPlayfield();
          State* state = aiSearch(tetrisBot.ai, tetrisBot.playfield, 
              tetrisBot.tetriminos);
          if (state != NULL) {
            buildStatesList(state, tetrisBot.states, &tetrisBot.statesLength);
            tetrisBot.movesIndex = 0;
            tetrisBot.moving = true;
          }
        }
      }
    }
  } else {
    tetrisBot.statesLength = 0;
    tetrisBot.moving = false;
    tetrisBot.playingDelay = 16;
  } 
}

void statusChanged(NintacoAPI* api, char* message) {
  puts(message);
}

void initTetrisBot() {
  tetrisBot.api = newNintacoAPI("localhost", 9999);
  tetrisBot.ai = newAI(NULL);
  tetrisBot.playfieldUtil = newPlayfieldUtil();
  tetrisBot.playfield = newPlayfield();
  tetrisBot.playingDelay = 0;
  tetrisBot.targetTetriminoY = 0;
  tetrisBot.startCounter = 0;
  tetrisBot.movesIndex = 0;
  tetrisBot.moving = false;
  tetrisBot.statesLength = 0;  
}

int main(int argc, char** argv) {
  initOrientations();
  initTetrisBot(); 
  
  addActivateListener(tetrisBot.api, &apiEnabled);
  addAccessPointListener(tetrisBot.api, &updateScore, PreExecute, 0x9C35);
  addAccessPointListener(tetrisBot.api, &speedUpDrop, PreExecute, 0x8977);
  addAccessPointListener(tetrisBot.api, &tetriminoYUpdated, PreWrite, 
      TetriminoY1);
  addAccessPointListener(tetrisBot.api, &tetriminoYUpdated, PreWrite, 
      TetriminoY2);
  addFrameListener(tetrisBot.api, &renderFinished);
  addStatusListener(tetrisBot.api, &statusChanged);
  run(tetrisBot.api);
  return EXIT_SUCCESS;
}