#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <stdarg.h>
#include <stdbool.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <unistd.h>

#ifdef WIN32
#include <windows.h>
#elif _POSIX_C_SOURCE >= 199309L
#include <time.h>   // for nanosleep
#else
#include <unistd.h> // for usleep
#endif

void sleep_ms(int milliseconds) // cross-platform sleep function
{
#ifdef WIN32
    Sleep(milliseconds);
#elif _POSIX_C_SOURCE >= 199309L
    struct timespec ts;
    ts.tv_sec = milliseconds / 1000;
    ts.tv_nsec = (milliseconds % 1000) * 1000000;
    nanosleep(&ts, NULL);
#else
    usleep(milliseconds * 1000);
#endif
}

#include "nintaco.h"

static const int ARRAY_LENGTH = 1024;

static const int EVENT_REQUEST = 0xFF;
static const int EVENT_RESPONSE = 0xFE;
static const int HEARTBEAT = 0xFD;
static const int READY = 0xFC;  
static const int RETRY_MILLIS = 1000;

typedef enum {
  Activate = 1,
  Deactivate = 3,
  Stop = 5,
  Access = 9,  
  Controllers = 11,  
  Frame = 13,  
  Scanline = 15,  
  ScanlineCycle = 17,  
  SpriteZero = 19,  
  Status = 21,
} EventTypes;

static const int EVENT_TYPES[] = { 
  Activate,
  Deactivate,
  Stop,
  Access,  
  Controllers,  
  Frame,  
  Scanline,  
  ScanlineCycle,  
  SpriteZero,  
  Status, 
};

typedef struct {  
  AccessPointListener listener;
  int type;
  int minAddress;
  int maxAddress;  
  int bank;
} AccessPoint;

static AccessPoint* newAccessPoint(AccessPointListener listener, 
    int type, int minAddress, int maxAddress, int bank) {
  
  AccessPoint* accessPoint = malloc(sizeof(AccessPoint));  
  accessPoint->listener = listener;
  accessPoint->type = type;
  accessPoint->bank = bank;
    
  if (maxAddress < 0) {
    accessPoint->minAddress = accessPoint->maxAddress = minAddress;
  } else if (minAddress <= maxAddress) {
    accessPoint->minAddress = minAddress;
    accessPoint->maxAddress = maxAddress;
  } else {
    accessPoint->minAddress = maxAddress;
    accessPoint->maxAddress = minAddress;
  }
  
  return accessPoint;
}

static void deleteAccessPoint(AccessPoint* accessPoint) {
  if (accessPoint != NULL) {
    free(accessPoint);
  }
}

typedef struct {
  ScanlineCycleListener listener;
  int scanline;
  int scanlineCycle;
} ScanlineCyclePoint;

static ScanlineCyclePoint* newScanlineCyclePoint(ScanlineCycleListener listener, 
    int scanline, int scanlineCycle) {  
  ScanlineCyclePoint* scanlineCyclePoint = (ScanlineCyclePoint*)malloc(
      sizeof(ScanlineCyclePoint));  
  scanlineCyclePoint->listener = listener;
  scanlineCyclePoint->scanline = scanline;
  scanlineCyclePoint->scanlineCycle = scanlineCycle;
  return scanlineCyclePoint;
}

static void deleteScanlineCyclePoint(ScanlineCyclePoint* scanlineCyclePoint) {
  if (scanlineCyclePoint != NULL) {
    free(scanlineCyclePoint);
  }
}

typedef struct {
  ScanlineListener listener;
  int scanline;  
} ScanlinePoint;

static ScanlinePoint* newScanlinePoint(ScanlineListener listener, int scanline){
  ScanlinePoint* scanlinePoint = (ScanlinePoint*)malloc(sizeof(ScanlinePoint));
  scanlinePoint->listener = listener;
  scanlinePoint->scanline = scanline;
  return scanlinePoint;
}

static void deleteScanlinePoint(ScanlinePoint* scanlinePoint) {
  if (scanlinePoint != NULL) {
    free(scanlinePoint);
  }
}

static DataOutputStream* newDataOutputStream(int fileDescriptor) {
  DataOutputStream* out = (DataOutputStream*)malloc(sizeof(DataOutputStream));
  out->size = 1024;
  out->head = 0;
  out->buffer = (unsigned char *)malloc(out->size 
      * sizeof(unsigned char));
  out->fileDescriptor = fileDescriptor;
  out->alive = true;
  return out;
}

static void deleteDataOutputStream(DataOutputStream* out) {
  if (out != NULL) {
    free(out->buffer);
    free(out);
  }
}

static void writeByte(DataOutputStream* out, unsigned char value) {
  if (out && out->alive) {
    if (out->head == out->size) {
      uint32_t size = out->size;
      unsigned char *buffer = out->buffer;
      out->size <<= 1;    
      out->buffer = (unsigned char *)malloc(out->size * sizeof(unsigned char));
      memcpy(out->buffer, buffer, size);
      free(buffer);
    }
    out->buffer[out->head] = value;
    out->head++;
  }
}

static void writeInt(DataOutputStream* out, int32_t value) {
  writeByte(out, (unsigned char)(value >> 24));
  writeByte(out, (unsigned char)(value >> 16));
  writeByte(out, (unsigned char)(value >> 8));
  writeByte(out, (unsigned char)(value));
}

static void writeIntArray(DataOutputStream* out, int32_t* array, int length) {
  int i;
  writeInt(out, length);
  for(i = 0; i < length; i++) {
    writeInt(out, array[i]);
  }
}

static void writeBoolean(DataOutputStream* out, bool value) {
  writeByte(out, (unsigned char)(value ? 1 : 0));
}

static void writeChar(DataOutputStream* out, char value) {
  writeByte(out, (unsigned char)value);
}

static void writeCharArray(DataOutputStream* out, char* array, int length) {
  int i;
  writeInt(out, length);
  for(i = 0; i < length; i++) {
    writeChar(out, array[i]);
  }
}

static void writeString(DataOutputStream* out, char* value) {
  writeCharArray(out, value, strlen(value));
}

static void writeStringArray(DataOutputStream* out, char** array, int length) {
  int i;
  writeInt(out, length);
  for(i = 0; i < length; i++) {
    writeString(out, array[i]);
  }
}

static void flush(DataOutputStream* out) {
  if (out->head > 0) {
    if (write(out->fileDescriptor, out->buffer, out->head) != out->head) {
      out->alive = false;
    }
    out->head = 0;
  }
}

static DataInputStream* newDataInputStream(int fileDescriptor) {
  DataInputStream* in = (DataInputStream*)malloc(sizeof(DataInputStream));
  in->capacity = 16 * 1024;
  in->size = 0;
  in->tail = 0;
  in->buffer = (unsigned char *)malloc(in->capacity * sizeof(unsigned char));
  in->fileDescriptor = fileDescriptor;
  in->alive = true;
  return in;
}

static void deleteDataInputStream(DataInputStream* in) {
  if (in != NULL) {
    free(in->buffer);
    free(in);
  }
}

static unsigned char readByte(DataInputStream* in) {
  while(in && in->alive && in->tail >= in->size) {
    in->tail = 0;
    in->size = read(in->fileDescriptor, in->buffer, in->capacity);
    if (in->size <= 0) {
      in->alive = false;      
    }
  }  
  if (in && in->alive) {
    return in->buffer[in->tail++];
  } else {
    return 0;
  }
}

static int32_t readInt(DataInputStream* in) {
  int32_t b1 = readByte(in);
  int32_t b2 = readByte(in);
  int32_t b3 = readByte(in);
  int32_t b4 = readByte(in);
  return (b1 << 24) | (b2 << 16) | (b3 << 8) | b4;
}

static int readIntArray(DataInputStream* in, int32_t* array, int length) {  
  if (length < 0) {
    in->alive = false;
  } else {
    int i;
    for(i = 0; i < length; i++) {
      array[i] = readInt(in);
    }
  }
  return length;
}

static bool readBoolean(DataInputStream* in) {
  return readByte(in) != 0;
}

static char readChar(DataInputStream* in) {
  return (char)readByte(in);
}

static int readCharArray(DataInputStream* in, char* array, int length) {
  if (length < 0) {
    in->alive = false;
  } else {
    int i;
    for(i = 0; i < length; i++) {
      array[i] = readChar(in);
    }
  }
  return length;
}

static char* readString(DataInputStream* in) {
  int32_t length = readInt(in);
  if (length < 0 || length > ARRAY_LENGTH) {
    in->alive = false;
    return NULL;
  } else {
    char* cs = (char*)malloc(sizeof(char) * (length + 1));
    int i;
    for(i = 0; i < length; i++) {
      cs[i] = readChar(in);
    }
    cs[length] = (char)0;
    return cs;
  }
}

static int readStringArray(DataInputStream* in, char** array, int arrayLength) {
  int32_t length = readInt(in);
  if (length < 0 || length > arrayLength) {
    in->alive = false;
  } else {
    int i;
    for(i = 0; i < length; i++) {
      array[i] = readString(in);
    }
  }
  return length;
}

static char** readDynamicStringArray(DataInputStream* in) {
  int32_t length = readInt(in);
  if (length < 0 || length > ARRAY_LENGTH) {
    in->alive = false;
    return NULL;
  } else {
    char** array = (char**)malloc(sizeof(char*) * length);
    int i;
    for(i = 0; i < length; i++) {
      array[i] = readString(in);
    }
    return array;
  }  
}

static ObjectIntEntry* newObjectIntEntry(void* key, int32_t value) {
  ObjectIntEntry* map = (ObjectIntEntry*)malloc(sizeof(ObjectIntEntry));
  map->key = key;
  map->value = value;
  map->next = NULL;
  return map;
}

static void deleteObjectIntEntry(ObjectIntEntry* map) {
  if (map != NULL) {
    deleteObjectIntEntry(map->next);
    free(map);
  }
}

static ObjectIntEntry* ObjectIntEntryClone(ObjectIntEntry* map) {
  if (map == NULL) {
    return NULL;
  } else {
    ObjectIntEntry* m = newObjectIntEntry(map->key, map->value);
    ObjectIntEntry* result = m;
    while(map->next) {
      map = map->next;
      m->next = newObjectIntEntry(map->key, map->value);
      m = m->next;
    }
    return result;
  }
}

static ObjectIntEntry* ObjectIntEntryGet(ObjectIntEntry* map, void* key) {
  while(!map && map->key != key) {
    map = map->next;
  }
  return map;
}

static ObjectIntEntry* ObjectIntEntryPut(ObjectIntEntry* map, void* key, 
    int32_t value) {
  
  if (map == NULL) {
    return newObjectIntEntry(key, value);
  } else {
    ObjectIntEntry* m = map;
    while(m->key != key) {
      if (m->next == NULL) {
        return (m->next = newObjectIntEntry(key, value));
      } else {
        m = m->next;
      }
    }
    m->value = value;
    return map;
  }
}

static ObjectIntEntry* ObjectIntEntryRemove(ObjectIntEntry* map, 
    ObjectIntEntry* entry) {
  
  ObjectIntEntry* result;
  
  if (map == NULL || entry == NULL) {
    result = map;
  } else if (map == entry) {
    result = map->next;
    free(map);    
  } else {
    result = map;
    while(map) {
      if (map->next == entry) {
        ObjectIntEntry* next = map->next->next;
        free(map->next);
        map->next = next;
        break;
      }
      map = map->next;
    }
  }
  
  return result;
}

static IntObjectEntry* newIntObjectEntry(int32_t key, void* value) {
  IntObjectEntry* map = (IntObjectEntry*)malloc(sizeof(IntObjectEntry));
  map->key = key;
  map->value = value;
  map->next = NULL;
  return map;
}

static void deleteIntObjectEntry(IntObjectEntry* map) {
  if (map != NULL) {
    deleteIntObjectEntry(map->next);
    free(map);
  }
}

static IntObjectEntry* IntObjectEntryClone(IntObjectEntry* map) {
  if (map == NULL) {
    return NULL;
  } else {
    IntObjectEntry* m = newIntObjectEntry(map->key, map->value);
    IntObjectEntry* result = m;
    while(map->next) {
      map = map->next;
      m->next = newIntObjectEntry(map->key, map->value);
      m = m->next;
    }
    return result;
  }
}

static IntObjectEntry* IntObjectEntryGet(IntObjectEntry* map, int32_t key) {
  while(!map && map->key != key) {
    map = map->next;
  }
  return map;
}

static IntObjectEntry* IntObjectEntryPut(IntObjectEntry* map, int32_t key, 
    void* value) {
  
  if (map == NULL) {
    return newIntObjectEntry(key, value);
  } else {
    IntObjectEntry* m = map;
    while(m->key != key) {
      if (m->next == NULL) {
        return (m->next = newIntObjectEntry(key, value));
      } else {
        m = m->next;
      }
    }
    m->value = value;
    return map;
  }
}

static IntObjectEntry* IntObjectEntryRemove(IntObjectEntry* map, 
    IntObjectEntry* entry) {
  
  IntObjectEntry* result;
  
  if (map == NULL || entry == NULL) {
    result = map;
  } else if (map == entry) {
    result = map->next;
    free(map);    
  } else {
    result = map;
    while(map) {
      if (map->next == entry) {
        IntObjectEntry* next = map->next->next;
        free(map->next);
        map->next = next;
        break;
      }
      map = map->next;
    }
  }
  
  return result;
}

static void fireDeactivated(NintacoAPI* api) {
  IntObjectEntry* map = IntObjectEntryClone(api->listenerObjects[Deactivate]);
  while(map) {
    ((DeactivateListener)map->value)(api);
    map = map->next;
  }
  deleteIntObjectEntry(map);
}

static void fireStatusChanged(NintacoAPI* api, char* message, ...) {
  char msg[256];
  va_list args;
  IntObjectEntry* map = IntObjectEntryClone(api->listenerObjects[Status]);
  
  va_start(args, message);
  vsprintf(msg, message, args);
  va_end(args);

  while(map) {
    ((StatusListener)map->value)(api, msg);
    map = map->next;
  }
  deleteIntObjectEntry(map);
}

static void sendReady(NintacoAPI* api) {
  DataOutputStream *out = api->out;
  if (out != NULL && out->alive) {
    writeByte(out, READY);
    flush(out);
  }
}

static void sendListener(NintacoAPI* api, int32_t listenerID, int32_t eventType, 
    void* listenerObject) {
  
  DataOutputStream *out = api->out;  
  if (out && out->alive) {    
    writeByte(out, eventType);
    writeInt(out, listenerID);
    switch(eventType) {
      case Access: {
        AccessPoint* point = (AccessPoint*)listenerObject;
        writeInt(out, point->type);
        writeInt(out, point->minAddress);
        writeInt(out, point->maxAddress);
        writeInt(out, point->bank);
        break;
      }
      case Scanline: {
        ScanlinePoint* point = (ScanlinePoint*)listenerObject;
        writeInt(out, point->scanline);
        break;
      }
      case ScanlineCycle: {
        ScanlineCyclePoint* point = (ScanlineCyclePoint*)listenerObject;
        writeInt(out, point->scanline);
        writeInt(out, point->scanlineCycle);
        break;
      }
    }
    flush(out);
  }
}

static void sendListeners(NintacoAPI* api) {
  IntObjectEntry** listenerObjects = api->listenerObjects;
  for(int i = 9; i >= 0; i--) {
    int32_t eventType = EVENT_TYPES[i];
    IntObjectEntry* map = listenerObjects[eventType];
    while(map) {
      sendListener(api, map->key, eventType, map->value);
      map = map->next;
    }
  }      
}

static int addListenerObject(NintacoAPI* api, void* listener,
    int32_t eventType, void* listenerObject) {  
  int32_t listenerID = api->nextID++;
  api->listenerIDs = ObjectIntEntryPut(api->listenerIDs, listener, listenerID);
  api->listenerObjects[eventType] = IntObjectEntryPut(
      api->listenerObjects[eventType], listenerID, listenerObject);
  return listenerID;
}

static int addListenerObj(NintacoAPI* api, void* listener, 
    int32_t eventType) {
  return addListenerObject(api, listener, eventType, listener);
}

static int removeListenerObject(NintacoAPI* api, void* listener, 
    int32_t eventType) {
  
  int32_t result;
  ObjectIntEntry* entry = ObjectIntEntryGet(api->listenerIDs, listener);
  if (entry == NULL) {
    result = -1;
  } else {
    result = entry->value;
    IntObjectEntry* e = IntObjectEntryGet(api->listenerObjects[eventType], 
        result);
    api->listenerObjects[eventType] = IntObjectEntryRemove(
        api->listenerObjects[eventType], e);
    api->listenerIDs = ObjectIntEntryRemove(api->listenerIDs, entry);
  }
  return result;
}

static void addListener(NintacoAPI* api, void* listener, int32_t eventType) {
  if (listener != NULL) {      
    sendListener(api, addListenerObj(api, listener, eventType), eventType, 
        listener);
  }
}

static void removeListener(NintacoAPI* api, void* listener, int32_t eventType, 
    int32_t methodValue) {
  if (listener != NULL) {
    int32_t listenerID = removeListenerObject(api, listener, eventType);
    DataOutputStream* out = api->out;
    if (listenerID >= 0 && out != NULL && out->alive) {
      writeByte(out, methodValue);
      writeInt(out, listenerID);
      flush(out);
    }
  }
}

static void probeEvents(NintacoAPI* api) {
  
  int32_t eventType;
  int32_t listenerID;
  IntObjectEntry* obj;
  IntObjectEntry** listenerObjects = api->listenerObjects;
  DataOutputStream *out = api->out;
  DataInputStream *in = api->in;
  if (out == NULL || in == NULL || !out->alive || !in->alive) {
    return;
  }

  writeByte(out, EVENT_REQUEST);
  flush(out);
  if (!out->alive || !in->alive) {
    return;
  }

  eventType = readByte(in);
  if (!out->alive || !in->alive) {
    return;
  }

  if (eventType == HEARTBEAT) {
    writeByte(out, EVENT_RESPONSE);
    flush(out);
    return;
  }
  
  listenerID = readInt(in);
  if (!out->alive || !in->alive) {
    return;
  }
  
  obj = IntObjectEntryGet(listenerObjects[eventType], listenerID);

  if (obj != NULL) {
    if (eventType == Access) {
      int32_t type = readInt(in);
      int32_t address = readInt(in);
      int32_t value = readInt(in);
      int32_t result = ((AccessPoint*)obj->value)->listener(api, type, address, 
          value);
      writeByte(out, EVENT_RESPONSE);
      writeInt(out, result);
    } else {
      switch(eventType) {
        case Activate:
          ((ActivateListener)obj->value)(api);
          break;
        case Deactivate:
          ((DeactivateListener)obj->value)(api);
          break;
        case Stop:
          ((StopListener)obj->value)(api);
          break;
        case Controllers:
          ((ControllersListener)obj->value)(api);
          break;
        case Frame:
          ((FrameListener)obj->value)(api);
          break;
        case Scanline:
          ((ScanlinePoint*)obj->value)->listener(api, readInt(in));
          break;
        case ScanlineCycle: {
          int32_t scanline = readInt(in);
          int32_t scanlineCycle = readInt(in);
          int32_t address = readInt(in); 
          bool rendering = readBoolean(in);
          ((ScanlineCyclePoint*)obj->value)->listener(api, scanline, 
              scanlineCycle, address, rendering);
          break;
        }
        case SpriteZero: {
          int32_t scanline = readInt(in);
          int32_t scanlineCycle = readInt(in);
          ((SpriteZeroListener)obj->value)(api, scanline, scanlineCycle);             
          break;
        }
        case Status:
          ((StatusListener)obj->value)(api, readString(in));
          break;
        default:
          out->alive = false;
          in->alive = false;
          break;
      } 
      writeByte(out, EVENT_RESPONSE);
    }
  }
  flush(out);
}

static void openSocketConnection(NintacoAPI* api) {
  
  struct addrinfo hints, *res;
  int fileDescriptor; 
  char port[32];
  
  sprintf(port, "%d", api->port);

  memset(&hints, 0, sizeof hints);
  hints.ai_family = AF_UNSPEC;
  hints.ai_socktype = SOCK_STREAM;

  getaddrinfo(api->host, port, &hints, &res);

  fileDescriptor = socket(res->ai_family, res->ai_socktype, res->ai_protocol);
  if (fileDescriptor < 0) {
    return;
  }
  if (connect(fileDescriptor, res->ai_addr, res->ai_addrlen) < 0) {
    return;
  }
  
  api->in = newDataInputStream(fileDescriptor);
  api->out = newDataOutputStream(fileDescriptor);  
}

void closeSocketConnect(NintacoAPI* api) {
  if (api->in != NULL || api->out != NULL) {
    close(api->in != NULL ? api->in->fileDescriptor : api->out->fileDescriptor);    
    deleteDataInputStream(api->in);
    deleteDataOutputStream(api->out);
    api->in = NULL;
    api->out = NULL;
  }
}

void run(NintacoAPI* api) {
  if (api->running) {
    return;
  } else {
    api->running = true;
  }
  while(true) {
    fireStatusChanged(api, "Connecting to %s:%d...", api->host, api->port);
    openSocketConnection(api);
    if (api->in == NULL) {
      fireStatusChanged(api, "Failed to establish connection.");
    } else {
      fireStatusChanged(api, "Connection established.");
      sendListeners(api);
      sendReady(api);
      while(api->in && api->out && api->in->alive && api->out->alive) {
        probeEvents(api);
      }
      closeSocketConnect(api);
      fireDeactivated(api);
      fireStatusChanged(api, "Disconnected.");      
    }
    sleep_ms(RETRY_MILLIS);
  }
}

NintacoAPI* newNintacoAPI(char*host, int port) {
  int i;
  NintacoAPI* api = (NintacoAPI*)malloc(sizeof(NintacoAPI));
  api->host = host;
  api->port = port;
  api->listenerIDs = NULL;
  api->listenerObjects = (IntObjectEntry**)malloc(sizeof(IntObjectEntry*) * 32);
  for(int i = 31; i >= 0; i--) {
    api->listenerObjects[i] = NULL;
  }
  api->nextID = 0;
  api->running = false;
  api->in = NULL;
  api->out = NULL;
  return api;
}

void deleteNintacoAPI(NintacoAPI* api) {
  if (api != NULL) {
    int i;  
    closeSocketConnect(api);
    deleteObjectIntEntry(api->listenerIDs);
    for(int i = 31; i >= 0; i--) {
      deleteIntObjectEntry(api->listenerObjects[i]);
    }
    free(api->listenerObjects);    
    free(api);
  }
}

void addActivateListener(NintacoAPI* api, ActivateListener listener) {
  addListener(api, listener, Activate);
}

void removeActivateListener(NintacoAPI* api, ActivateListener listener) {
  removeListener(api, listener, Activate, 2);
}

void addDeactivateListener(NintacoAPI* api, DeactivateListener listener) {
  addListener(api, listener, Deactivate);
}

void removeDeactivateListener(NintacoAPI* api, DeactivateListener listener) {
  removeListener(api, listener, Deactivate, 4);
}

void addStopListener(NintacoAPI* api, StopListener listener) {
  addListener(api, listener, Stop);
}

void removeStopListener(NintacoAPI* api, StopListener listener) {
  removeListener(api, listener, Stop, 6);
}

void addAccessPointListener3(NintacoAPI* api, AccessPointListener listener, 
    int32_t accessPointType, int32_t minAddress, int32_t maxAddress, 
        int32_t bank) {

  if (listener != NULL) {
    AccessPoint* point = newAccessPoint(listener, accessPointType, 
        minAddress, maxAddress, bank);
    sendListener(api, addListenerObject(api, listener, Access, point), Access, 
        point);
  }
}

void addAccessPointListener2(NintacoAPI* api, AccessPointListener listener, 
    int32_t accessPointType, int32_t minAddress, int32_t maxAddress) {    
  addAccessPointListener3(api, listener, accessPointType, minAddress, 
      maxAddress, -1);
}

void addAccessPointListener(NintacoAPI* api, AccessPointListener listener, 
    int32_t accessPointType, int32_t address) {    
  addAccessPointListener3(api, listener, accessPointType, address, -1, -1);
}

void removeAccessPointListener(NintacoAPI* api, AccessPointListener listener) {
  removeListener(api, listener, Access, 10);
}

void addControllersListener(NintacoAPI* api, ControllersListener listener) {
  addListener(api, listener, Controllers);
}

void removeControllersListener(NintacoAPI* api, ControllersListener listener) {
  removeListener(api, listener, Controllers, 12);
}

void addFrameListener(NintacoAPI* api, FrameListener listener) {
  addListener(api, listener, Frame);
}

void removeFrameListener(NintacoAPI* api, FrameListener listener) {
  removeListener(api, listener, Frame, 14);
}  
  
void addScanlineListener(NintacoAPI* api, ScanlineListener listener, 
    int32_t scanline) {   

  if (listener != NULL) {
    ScanlinePoint* point = newScanlinePoint(listener, scanline);
    sendListener(api, addListenerObject(api, listener, Scanline, point), 
        Scanline, point);
  }
}

void removeScanlineListener(NintacoAPI* api, ScanlineListener listener) {
  removeListener(api, listener, Scanline, 16);
}  
  
void addScanlineCycleListener(NintacoAPI* api, ScanlineCycleListener listener,
    int32_t scanline, int32_t scanlineCycle) {

  if (listener != NULL) {
    ScanlineCyclePoint* point = newScanlineCyclePoint(listener, 
        scanline, scanlineCycle);
    sendListener(api, addListenerObject(api, listener, ScanlineCycle, point), 
        ScanlineCycle, point);
  }
}

void removeScanlineCycleListener(NintacoAPI* api, 
    ScanlineCycleListener listener) {
  removeListener(api, listener, ScanlineCycle, 18);
}
  
void addSpriteZeroListener(NintacoAPI* api, SpriteZeroListener listener) {
  addListener(api, listener, SpriteZero);
}

void removeSpriteZeroListener(NintacoAPI* api, SpriteZeroListener listener) {
  removeListener(api, listener, SpriteZero, 20);
}

void addStatusListener(NintacoAPI* api, StatusListener listener) {
  addListener(api, listener, Status);
}

void removeStatusListener(NintacoAPI* api, StatusListener listener) {
  removeListener(api, listener, Status, 22);
}

void getPixels(NintacoAPI* api, int* pixels) {
  if (api != NULL && pixels != NULL) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out != NULL && in != NULL && out->alive && in->alive) {
      writeByte(out, 119);
      flush(out);
      readIntArray(in, pixels, readInt(in));
    }
  }
}

/* AUTOGENERATED CONTENT FOLLOWS. DO NOT MODIFY. */

void setPaused(NintacoAPI* api, bool paused) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 23);
      writeBoolean(out, paused);
      flush(out);
    }
  }
}

bool isPaused(NintacoAPI* api) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 24);
      flush(out);
      return readBoolean(in);
    }
  }
  return false;
}

int getFrameCount(NintacoAPI* api) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 25);
      flush(out);
      return readInt(in);
    }
  }
  return -1;
}

int getA(NintacoAPI* api) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 26);
      flush(out);
      return readInt(in);
    }
  }
  return -1;
}

void setA(NintacoAPI* api, int A) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 27);
      writeInt(out, A);
      flush(out);
    }
  }
}

int getS(NintacoAPI* api) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 28);
      flush(out);
      return readInt(in);
    }
  }
  return -1;
}

void setS(NintacoAPI* api, int S) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 29);
      writeInt(out, S);
      flush(out);
    }
  }
}

int getPC(NintacoAPI* api) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 30);
      flush(out);
      return readInt(in);
    }
  }
  return -1;
}

void setPC(NintacoAPI* api, int PC) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 31);
      writeInt(out, PC);
      flush(out);
    }
  }
}

int getX(NintacoAPI* api) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 32);
      flush(out);
      return readInt(in);
    }
  }
  return -1;
}

void setX(NintacoAPI* api, int X) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 33);
      writeInt(out, X);
      flush(out);
    }
  }
}

int getY(NintacoAPI* api) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 34);
      flush(out);
      return readInt(in);
    }
  }
  return -1;
}

void setY(NintacoAPI* api, int Y) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 35);
      writeInt(out, Y);
      flush(out);
    }
  }
}

int getP(NintacoAPI* api) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 36);
      flush(out);
      return readInt(in);
    }
  }
  return -1;
}

void setP(NintacoAPI* api, int P) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 37);
      writeInt(out, P);
      flush(out);
    }
  }
}

bool isN(NintacoAPI* api) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 38);
      flush(out);
      return readBoolean(in);
    }
  }
  return false;
}

void setN(NintacoAPI* api, bool N) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 39);
      writeBoolean(out, N);
      flush(out);
    }
  }
}

bool isV(NintacoAPI* api) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 40);
      flush(out);
      return readBoolean(in);
    }
  }
  return false;
}

void setV(NintacoAPI* api, bool V) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 41);
      writeBoolean(out, V);
      flush(out);
    }
  }
}

bool isD(NintacoAPI* api) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 42);
      flush(out);
      return readBoolean(in);
    }
  }
  return false;
}

void setD(NintacoAPI* api, bool D) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 43);
      writeBoolean(out, D);
      flush(out);
    }
  }
}

bool isI(NintacoAPI* api) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 44);
      flush(out);
      return readBoolean(in);
    }
  }
  return false;
}

void setI(NintacoAPI* api, bool I) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 45);
      writeBoolean(out, I);
      flush(out);
    }
  }
}

bool isZ(NintacoAPI* api) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 46);
      flush(out);
      return readBoolean(in);
    }
  }
  return false;
}

void setZ(NintacoAPI* api, bool Z) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 47);
      writeBoolean(out, Z);
      flush(out);
    }
  }
}

bool isC(NintacoAPI* api) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 48);
      flush(out);
      return readBoolean(in);
    }
  }
  return false;
}

void setC(NintacoAPI* api, bool C) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 49);
      writeBoolean(out, C);
      flush(out);
    }
  }
}

int getPPUv(NintacoAPI* api) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 50);
      flush(out);
      return readInt(in);
    }
  }
  return -1;
}

void setPPUv(NintacoAPI* api, int v) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 51);
      writeInt(out, v);
      flush(out);
    }
  }
}

int getPPUt(NintacoAPI* api) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 52);
      flush(out);
      return readInt(in);
    }
  }
  return -1;
}

void setPPUt(NintacoAPI* api, int t) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 53);
      writeInt(out, t);
      flush(out);
    }
  }
}

int getPPUx(NintacoAPI* api) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 54);
      flush(out);
      return readInt(in);
    }
  }
  return -1;
}

void setPPUx(NintacoAPI* api, int x) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 55);
      writeInt(out, x);
      flush(out);
    }
  }
}

bool isPPUw(NintacoAPI* api) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 56);
      flush(out);
      return readBoolean(in);
    }
  }
  return false;
}

void setPPUw(NintacoAPI* api, bool w) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 57);
      writeBoolean(out, w);
      flush(out);
    }
  }
}

int getCameraX(NintacoAPI* api) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 58);
      flush(out);
      return readInt(in);
    }
  }
  return -1;
}

void setCameraX(NintacoAPI* api, int scrollX) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 59);
      writeInt(out, scrollX);
      flush(out);
    }
  }
}

int getCameraY(NintacoAPI* api) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 60);
      flush(out);
      return readInt(in);
    }
  }
  return -1;
}

void setCameraY(NintacoAPI* api, int scrollY) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 61);
      writeInt(out, scrollY);
      flush(out);
    }
  }
}

int getScanline(NintacoAPI* api) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 62);
      flush(out);
      return readInt(in);
    }
  }
  return -1;
}

int getDot(NintacoAPI* api) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 63);
      flush(out);
      return readInt(in);
    }
  }
  return -1;
}

bool isSpriteZeroHit(NintacoAPI* api) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 64);
      flush(out);
      return readBoolean(in);
    }
  }
  return false;
}

void setSpriteZeroHit(NintacoAPI* api, bool sprite0Hit) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 65);
      writeBoolean(out, sprite0Hit);
      flush(out);
    }
  }
}

int getScanlineCount(NintacoAPI* api) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 66);
      flush(out);
      return readInt(in);
    }
  }
  return -1;
}

void requestInterrupt(NintacoAPI* api) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 67);
      flush(out);
    }
  }
}

void acknowledgeInterrupt(NintacoAPI* api) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 68);
      flush(out);
    }
  }
}

int peekCPU(NintacoAPI* api, int address) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 69);
      writeInt(out, address);
      flush(out);
      return readInt(in);
    }
  }
  return -1;
}

int readCPU(NintacoAPI* api, int address) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 70);
      writeInt(out, address);
      flush(out);
      return readInt(in);
    }
  }
  return -1;
}

void writeCPU(NintacoAPI* api, int address, int value) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 71);
      writeInt(out, address);
      writeInt(out, value);
      flush(out);
    }
  }
}

int peekCPU16(NintacoAPI* api, int address) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 72);
      writeInt(out, address);
      flush(out);
      return readInt(in);
    }
  }
  return -1;
}

int readCPU16(NintacoAPI* api, int address) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 73);
      writeInt(out, address);
      flush(out);
      return readInt(in);
    }
  }
  return -1;
}

void writeCPU16(NintacoAPI* api, int address, int value) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 74);
      writeInt(out, address);
      writeInt(out, value);
      flush(out);
    }
  }
}

int peekCPU32(NintacoAPI* api, int address) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 75);
      writeInt(out, address);
      flush(out);
      return readInt(in);
    }
  }
  return -1;
}

int readCPU32(NintacoAPI* api, int address) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 76);
      writeInt(out, address);
      flush(out);
      return readInt(in);
    }
  }
  return -1;
}

void writeCPU32(NintacoAPI* api, int address, int value) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 77);
      writeInt(out, address);
      writeInt(out, value);
      flush(out);
    }
  }
}

int readPPU(NintacoAPI* api, int address) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 78);
      writeInt(out, address);
      flush(out);
      return readInt(in);
    }
  }
  return -1;
}

void writePPU(NintacoAPI* api, int address, int value) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 79);
      writeInt(out, address);
      writeInt(out, value);
      flush(out);
    }
  }
}

int readPaletteRAM(NintacoAPI* api, int address) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 80);
      writeInt(out, address);
      flush(out);
      return readInt(in);
    }
  }
  return -1;
}

void writePaletteRAM(NintacoAPI* api, int address, int value) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 81);
      writeInt(out, address);
      writeInt(out, value);
      flush(out);
    }
  }
}

int readOAM(NintacoAPI* api, int address) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 82);
      writeInt(out, address);
      flush(out);
      return readInt(in);
    }
  }
  return -1;
}

void writeOAM(NintacoAPI* api, int address, int value) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 83);
      writeInt(out, address);
      writeInt(out, value);
      flush(out);
    }
  }
}

bool readGamepad(NintacoAPI* api, int gamepad, int button) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 84);
      writeInt(out, gamepad);
      writeInt(out, button);
      flush(out);
      return readBoolean(in);
    }
  }
  return false;
}

void writeGamepad(NintacoAPI* api, int gamepad, int button, bool value) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 85);
      writeInt(out, gamepad);
      writeInt(out, button);
      writeBoolean(out, value);
      flush(out);
    }
  }
}

bool isZapperTrigger(NintacoAPI* api) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 86);
      flush(out);
      return readBoolean(in);
    }
  }
  return false;
}

void setZapperTrigger(NintacoAPI* api, bool zapperTrigger) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 87);
      writeBoolean(out, zapperTrigger);
      flush(out);
    }
  }
}

int getZapperX(NintacoAPI* api) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 88);
      flush(out);
      return readInt(in);
    }
  }
  return -1;
}

void setZapperX(NintacoAPI* api, int x) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 89);
      writeInt(out, x);
      flush(out);
    }
  }
}

int getZapperY(NintacoAPI* api) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 90);
      flush(out);
      return readInt(in);
    }
  }
  return -1;
}

void setZapperY(NintacoAPI* api, int y) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 91);
      writeInt(out, y);
      flush(out);
    }
  }
}

void setColor(NintacoAPI* api, int color) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 92);
      writeInt(out, color);
      flush(out);
    }
  }
}

int getColor(NintacoAPI* api) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 93);
      flush(out);
      return readInt(in);
    }
  }
  return -1;
}

void setClip(NintacoAPI* api, int x, int y, int width, int height) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 94);
      writeInt(out, x);
      writeInt(out, y);
      writeInt(out, width);
      writeInt(out, height);
      flush(out);
    }
  }
}

void clipRect(NintacoAPI* api, int x, int y, int width, int height) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 95);
      writeInt(out, x);
      writeInt(out, y);
      writeInt(out, width);
      writeInt(out, height);
      flush(out);
    }
  }
}

void resetClip(NintacoAPI* api) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 96);
      flush(out);
    }
  }
}

void copyArea(NintacoAPI* api, int x, int y, int width, int height, int dx, 
      int dy) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 97);
      writeInt(out, x);
      writeInt(out, y);
      writeInt(out, width);
      writeInt(out, height);
      writeInt(out, dx);
      writeInt(out, dy);
      flush(out);
    }
  }
}

void drawLine(NintacoAPI* api, int x1, int y1, int x2, int y2) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 98);
      writeInt(out, x1);
      writeInt(out, y1);
      writeInt(out, x2);
      writeInt(out, y2);
      flush(out);
    }
  }
}

void drawOval(NintacoAPI* api, int x, int y, int width, int height) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 99);
      writeInt(out, x);
      writeInt(out, y);
      writeInt(out, width);
      writeInt(out, height);
      flush(out);
    }
  }
}

void drawPolygon(NintacoAPI* api, int* xPoints, int* yPoints, int nPoints) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 100);
      writeIntArray(out, xPoints, nPoints);
      writeIntArray(out, yPoints, nPoints);
      writeInt(out, nPoints);
      flush(out);
    }
  }
}

void drawPolyline(NintacoAPI* api, int* xPoints, int* yPoints, int nPoints) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 101);
      writeIntArray(out, xPoints, nPoints);
      writeIntArray(out, yPoints, nPoints);
      writeInt(out, nPoints);
      flush(out);
    }
  }
}

void drawRect(NintacoAPI* api, int x, int y, int width, int height) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 102);
      writeInt(out, x);
      writeInt(out, y);
      writeInt(out, width);
      writeInt(out, height);
      flush(out);
    }
  }
}

void drawRoundRect(NintacoAPI* api, int x, int y, int width, int height, 
      int arcWidth, int arcHeight) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 103);
      writeInt(out, x);
      writeInt(out, y);
      writeInt(out, width);
      writeInt(out, height);
      writeInt(out, arcWidth);
      writeInt(out, arcHeight);
      flush(out);
    }
  }
}

void draw3DRect(NintacoAPI* api, int x, int y, int width, int height, 
      bool raised) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 104);
      writeInt(out, x);
      writeInt(out, y);
      writeInt(out, width);
      writeInt(out, height);
      writeBoolean(out, raised);
      flush(out);
    }
  }
}

void drawArc(NintacoAPI* api, int x, int y, int width, int height, 
      int startAngle, int arcAngle) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 105);
      writeInt(out, x);
      writeInt(out, y);
      writeInt(out, width);
      writeInt(out, height);
      writeInt(out, startAngle);
      writeInt(out, arcAngle);
      flush(out);
    }
  }
}

void fill3DRect(NintacoAPI* api, int x, int y, int width, int height, 
      bool raised) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 106);
      writeInt(out, x);
      writeInt(out, y);
      writeInt(out, width);
      writeInt(out, height);
      writeBoolean(out, raised);
      flush(out);
    }
  }
}

void fillArc(NintacoAPI* api, int x, int y, int width, int height, 
      int startAngle, int arcAngle) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 107);
      writeInt(out, x);
      writeInt(out, y);
      writeInt(out, width);
      writeInt(out, height);
      writeInt(out, startAngle);
      writeInt(out, arcAngle);
      flush(out);
    }
  }
}

void fillOval(NintacoAPI* api, int x, int y, int width, int height) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 108);
      writeInt(out, x);
      writeInt(out, y);
      writeInt(out, width);
      writeInt(out, height);
      flush(out);
    }
  }
}

void fillPolygon(NintacoAPI* api, int* xPoints, int* yPoints, int nPoints) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 109);
      writeIntArray(out, xPoints, nPoints);
      writeIntArray(out, yPoints, nPoints);
      writeInt(out, nPoints);
      flush(out);
    }
  }
}

void fillRect(NintacoAPI* api, int x, int y, int width, int height) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 110);
      writeInt(out, x);
      writeInt(out, y);
      writeInt(out, width);
      writeInt(out, height);
      flush(out);
    }
  }
}

void fillRoundRect(NintacoAPI* api, int x, int y, int width, int height, 
      int arcWidth, int arcHeight) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 111);
      writeInt(out, x);
      writeInt(out, y);
      writeInt(out, width);
      writeInt(out, height);
      writeInt(out, arcWidth);
      writeInt(out, arcHeight);
      flush(out);
    }
  }
}

void drawChar(NintacoAPI* api, char c, int x, int y) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 112);
      writeChar(out, c);
      writeInt(out, x);
      writeInt(out, y);
      flush(out);
    }
  }
}

void drawChars(NintacoAPI* api, char* data, int offset, int length, int x, 
      int y, bool monospaced) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 113);
      writeCharArray(out, data, offset + length);
      writeInt(out, offset);
      writeInt(out, length);
      writeInt(out, x);
      writeInt(out, y);
      writeBoolean(out, monospaced);
      flush(out);
    }
  }
}

void drawString(NintacoAPI* api, char* str, int x, int y, bool monospaced) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 114);
      writeString(out, str);
      writeInt(out, x);
      writeInt(out, y);
      writeBoolean(out, monospaced);
      flush(out);
    }
  }
}

void createSprite(NintacoAPI* api, int id, int width, int height, int* pixels) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 115);
      writeInt(out, id);
      writeInt(out, width);
      writeInt(out, height);
      writeIntArray(out, pixels, width * height);
      flush(out);
    }
  }
}

void drawSprite(NintacoAPI* api, int id, int x, int y) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 116);
      writeInt(out, id);
      writeInt(out, x);
      writeInt(out, y);
      flush(out);
    }
  }
}

void setPixel(NintacoAPI* api, int x, int y, int color) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 117);
      writeInt(out, x);
      writeInt(out, y);
      writeInt(out, color);
      flush(out);
    }
  }
}

int getPixel(NintacoAPI* api, int x, int y) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 118);
      writeInt(out, x);
      writeInt(out, y);
      flush(out);
      return readInt(in);
    }
  }
  return -1;
}

void powerCycle(NintacoAPI* api) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 120);
      flush(out);
    }
  }
}

void reset(NintacoAPI* api) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 121);
      flush(out);
    }
  }
}

void deleteSprite(NintacoAPI* api, int id) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 122);
      writeInt(out, id);
      flush(out);
    }
  }
}

void setSpeed(NintacoAPI* api, int percent) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 123);
      writeInt(out, percent);
      flush(out);
    }
  }
}

void stepToNextFrame(NintacoAPI* api) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 124);
      flush(out);
    }
  }
}

void showMessage(NintacoAPI* api, char* message) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 125);
      writeString(out, message);
      flush(out);
    }
  }
}

char* getWorkingDirectory(NintacoAPI* api) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 126);
      flush(out);
      return readString(in);
    }
  }
  return NULL;
}

char* getContentDirectory(NintacoAPI* api) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 127);
      flush(out);
      return readString(in);
    }
  }
  return NULL;
}

void openFile(NintacoAPI* api, char* fileName) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 128);
      writeString(out, fileName);
      flush(out);
    }
  }
}

void openArchiveEntry(NintacoAPI* api, char* archiveFileName, 
      char* entryFileName) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 129);
      writeString(out, archiveFileName);
      writeString(out, entryFileName);
      flush(out);
    }
  }
}

char** getArchiveEntries(NintacoAPI* api, char* archiveFileName) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 130);
      writeString(out, archiveFileName);
      flush(out);
      return readDynamicStringArray(in);
    }
  }
  return NULL;
}

char* getDefaultArchiveEntry(NintacoAPI* api, char* archiveFileName) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 131);
      writeString(out, archiveFileName);
      flush(out);
      return readString(in);
    }
  }
  return NULL;
}

void openDefaultArchiveEntry(NintacoAPI* api, char* archiveFileName) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 132);
      writeString(out, archiveFileName);
      flush(out);
    }
  }
}

void closeFile(NintacoAPI* api) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 133);
      flush(out);
    }
  }
}

void saveState(NintacoAPI* api, char* stateFileName) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 134);
      writeString(out, stateFileName);
      flush(out);
    }
  }
}

void loadState(NintacoAPI* api, char* stateFileName) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 135);
      writeString(out, stateFileName);
      flush(out);
    }
  }
}

void quickSaveState(NintacoAPI* api, int slot) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 136);
      writeInt(out, slot);
      flush(out);
    }
  }
}

void quickLoadState(NintacoAPI* api, int slot) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 137);
      writeInt(out, slot);
      flush(out);
    }
  }
}

void setTVSystem(NintacoAPI* api, char* tvSystem) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 138);
      writeString(out, tvSystem);
      flush(out);
    }
  }
}

char* getTVSystem(NintacoAPI* api) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 139);
      flush(out);
      return readString(in);
    }
  }
  return NULL;
}

int getDiskSides(NintacoAPI* api) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 140);
      flush(out);
      return readInt(in);
    }
  }
  return -1;
}

void insertDisk(NintacoAPI* api, int disk, int side) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 141);
      writeInt(out, disk);
      writeInt(out, side);
      flush(out);
    }
  }
}

void flipDiskSide(NintacoAPI* api) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 142);
      flush(out);
    }
  }
}

void ejectDisk(NintacoAPI* api) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 143);
      flush(out);
    }
  }
}

void insertCoin(NintacoAPI* api) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 144);
      flush(out);
    }
  }
}

void pressServiceButton(NintacoAPI* api) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 145);
      flush(out);
    }
  }
}

void screamIntoMicrophone(NintacoAPI* api) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 146);
      flush(out);
    }
  }
}

void glitch(NintacoAPI* api) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 147);
      flush(out);
    }
  }
}

char* getFileInfo(NintacoAPI* api) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 148);
      flush(out);
      return readString(in);
    }
  }
  return NULL;
}

void setFullscreenMode(NintacoAPI* api, bool fullscreenMode) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 149);
      writeBoolean(out, fullscreenMode);
      flush(out);
    }
  }
}

void saveScreenshot(NintacoAPI* api) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 150);
      flush(out);
    }
  }
}

void addCheat(NintacoAPI* api, int address, int value, int compare, 
      char* description, bool enabled) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 151);
      writeInt(out, address);
      writeInt(out, value);
      writeInt(out, compare);
      writeString(out, description);
      writeBoolean(out, enabled);
      flush(out);
    }
  }
}

void removeCheat(NintacoAPI* api, int address, int value, int compare) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 152);
      writeInt(out, address);
      writeInt(out, value);
      writeInt(out, compare);
      flush(out);
    }
  }
}

void addGameGenie(NintacoAPI* api, char* gameGenieCode, char* description, 
      bool enabled) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 153);
      writeString(out, gameGenieCode);
      writeString(out, description);
      writeBoolean(out, enabled);
      flush(out);
    }
  }
}

void removeGameGenie(NintacoAPI* api, char* gameGenieCode) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 154);
      writeString(out, gameGenieCode);
      flush(out);
    }
  }
}

void addProActionRocky(NintacoAPI* api, char* proActionRockyCode, 
      char* description, bool enabled) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 155);
      writeString(out, proActionRockyCode);
      writeString(out, description);
      writeBoolean(out, enabled);
      flush(out);
    }
  }
}

void removeProActionRocky(NintacoAPI* api, char* proActionRockyCode) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 156);
      writeString(out, proActionRockyCode);
      flush(out);
    }
  }
}

int getPrgRomSize(NintacoAPI* api) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 157);
      flush(out);
      return readInt(in);
    }
  }
  return -1;
}

int readPrgRom(NintacoAPI* api, int index) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 158);
      writeInt(out, index);
      flush(out);
      return readInt(in);
    }
  }
  return -1;
}

void writePrgRom(NintacoAPI* api, int index, int value) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 159);
      writeInt(out, index);
      writeInt(out, value);
      flush(out);
    }
  }
}

int getChrRomSize(NintacoAPI* api) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 160);
      flush(out);
      return readInt(in);
    }
  }
  return -1;
}

int readChrRom(NintacoAPI* api, int index) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 161);
      writeInt(out, index);
      flush(out);
      return readInt(in);
    }
  }
  return -1;
}

void writeChrRom(NintacoAPI* api, int index, int value) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 162);
      writeInt(out, index);
      writeInt(out, value);
      flush(out);
    }
  }
}

int getStringWidth(NintacoAPI* api, char* str, bool monospaced) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 163);
      writeString(out, str);
      writeBoolean(out, monospaced);
      flush(out);
      return readInt(in);
    }
  }
  return -1;
}

int getCharsWidth(NintacoAPI* api, char* chars, bool monospaced) {
  if (api) {
    DataOutputStream *out = api->out;
    DataInputStream *in = api->in;
    if (out && in && out->alive && in->alive) {
      writeByte(out, 164);
      writeCharArray(out, chars, strlen(chars));
      writeBoolean(out, monospaced);
      flush(out);
      return readInt(in);
    }
  }
  return -1;
}